<?php

/**
 * Record file for KSeF integration model.
 *
 * @package   Settings.Model
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

use App\Cache;
use App\Db;
use App\Db\Query;
use App\Exceptions\IllegalValue;
use App\Fields\File;
use App\Integrations\KSeF\KSeF;
use App\Integrations\KSeF\Model\Certificate\Certificate;
use App\Integrations\KSeF\Model\Certificate\Enum\CertificateStatus;
use App\Integrations\KSeF\Model\Connector\Setting;
use App\Integrations\KSeF\Service\Connector\ConnectorService;
use App\Json;
use App\Language;
use App\Log;
use App\Privilege;
use App\Record;
use App\Request;
use N1ebieski\KSEFClient\Factories\EncryptionKeyFactory;

/**
 * Record class for KSeF integration model.
 */
class Settings_KSeF_Record_Model extends Settings_Vtiger_Record_Model
{
	public static $foo = [];
	/** @var array Record changes */
	protected array $changes = [];
	/** @var ConnectorService|null Connector service instance */
	private ?ConnectorService $connectorService = null;

	/** {@inheritdoc} */
	public function getId(): mixed
	{
		return $this->get('id');
	}

	/**
	 * Check if record is new.
	 *
	 * @return bool
	 */
	public function isNew()
	{
		return !$this->getId();
	}

	/** {@inheritdoc} */
	public function getName(): string
	{
		return $this->get('name') ?? '';
	}

	/**
	 * Function to get the instance of record model.
	 *
	 * @param int $id
	 *
	 * @return self|null instance, if exists
	 */
	public static function getInstanceById(int $id): ?self
	{
		$data = (new Query())
			->from(KSeF::SETTINGS_TABLE_NAME)
			->where(['id' => $id])
			->one();

		$instance = null;
		if ($data) {
			$instance = new self();
			$instance->setData($data);
		}

		return $instance;
	}

	/**
	 * Function to get the clean instance.
	 *
	 * @return self
	 */
	public static function getCleanInstance(): self
	{
		$instance = new self();
		$instance->getModule();

		return $instance;
	}

	/**
	 * Function to get Module instance.
	 *
	 * @return Settings_KSeF_Module_Model
	 */
	public function getModule(): Settings_KSeF_Module_Model
	{
		if (!isset($this->module)) {
			$this->module = Settings_KSeF_Module_Model::getInstance('Settings:KSeF');
		}

		return $this->module;
	}

	/**
	 * Function determines fields available in edition view.
	 *
	 * @param mixed  $name
	 * @param string $key
	 *
	 * @return Vtiger_Field_Model
	 */

	/** {@inheritdoc} */
	public function getDisplayValue(string $key)
	{
		$value = $this->get($key);
		$fieldModel = $this->getField($key);
		return match ($key) {
			'company' => Record::getLabel($value),
			default => $fieldModel->getUITypeModel()->getDisplayValue($value, $this->getId(), $this),
		};
	}

	/**
	 * Sets data from request.
	 *
	 * @param Request $request
	 */
	public function setDataFromRequest(Request $request): void
	{
		foreach ($this->getModule()->getFormFields() as $fieldName) {
			if ($request->has($fieldName)) {
				$fieldModel = $this->getField($fieldName);
				$fieldModelUIType = $fieldModel->getUITypeModel();
				if (330 === $fieldModel->getUIType()) {
					$value = $request->getArray($fieldName, $fieldModel->get('purifyType'));
					$previousValue = $this->get($fieldName);
					$previousValue = ($previousValue && !Json::isEmpty($previousValue)) ? File::parse(Json::decode($previousValue)) : [];
					[$value] = $fieldModelUIType->updateUploadFiles($value, $previousValue);
				} else {
					$value = $request->isEmpty($fieldName) && !$fieldModel->isMandatory() ? '' : $request->getByType($fieldName, $fieldModel->get('purifyType'));
				}
				$fieldModel->getUITypeModel()->validate($value, true);
				$value = $fieldModel->getUITypeModel()->getDBValue($value);
				$this->set($fieldName, $value);
			}
		}
	}

	public function getField(string $fieldName): ?Vtiger_Field_Model
	{
		$fieldModel = $this->getModule()->getFieldInstanceByName($fieldName);
		$fieldModel->set(
			'fieldvalue',
			($this->has($fieldName) && ($defaultValue = $fieldModel->get('defaultvalue'))) ? $defaultValue : ($this->get($fieldName) ?? '')
		);

		return $fieldModel;
	}

	/**
	 * Function to save.
	 *
	 * @throws IllegalValue
	 *
	 * @return bool
	 */
	public function save()
	{
		$db = Db::getInstance('admin');
		$transaction = $db->beginTransaction();
		try {
			$this->validate();
			$this->beforeSaveEvents();
			$this->saveToDb();
			$transaction->commit();
			$this->afterSaveEvents();
		} catch (Throwable $ex) {
			$transaction->rollBack();
			Log::error($ex->__toString());
			throw $ex;
		}
		$this->clearCache();
		return true;
	}

	/**
	 * Save data to the database.
	 */
	public function saveToDb()
	{
		$db = Db::getInstance('admin');
		$tablesData = $this->getId() ? array_intersect_key($this->getData(), $this->changes) : $this->getData();
		if ($tablesData) {
			$baseTable = $this->getModule()->baseTable;
			$baseTableIndex = $this->getModule()->baseIndex;
			if ($this->getId()) {
				$db->createCommand()->update($baseTable, $tablesData, [$baseTableIndex => (int) $this->getId()])->execute();
			} else {
				$db->createCommand()->insert($baseTable, $tablesData)->execute();
				$this->set('id', $db->getLastInsertID("{$baseTable}_{$baseTableIndex}_seq"));
			}
		}
	}

	/**
	 * Clear cache.
	 */
	public function clearCache()
	{
		Cache::delete('KSeF', '');
	}

	public function set($key, $value)
	{
		if ($this->getId() && !\in_array($key, ['id']) && (\array_key_exists($key, $this->value) && $this->value[$key] != $value) && !\array_key_exists($key, $this->changes)) {
			$this->changes[$key] = $this->get($key);
		}
		return parent::set($key, $value);
	}

	public function beforeSaveEvents(): void
	{
		if ($this->isNew() || false !== $this->getPreviousValue()) {
			$this->updateStatus();
		}
	}

	public function updateStatus()
	{
		$certPath = Json::decode($this->get('cert'))[0]['path'] ?? '';
		$certificate = Certificate::fromFile($certPath);
		$this->set('date_valid_from', $certificate->getValidFromDate());
		$this->set('date_valid_to', $certificate->getValidToDate());
		$status = $this->getConnection()->requestCertificateDetails($certificate)?->object()?->certificates[0]?->status ?? '';
		$this->set('status', CertificateStatus::tryFrom($status)?->getLabel());
	}

	public function afterSaveEvents(): void
	{
		foreach (['cert', 'key'] as $fieldName) {
			$currentData = [];
			if ($this->get($fieldName) && ($this->isNew() || false !== $this->getPreviousValue($fieldName))) {
				$currentData = File::parse(Json::decode($this->get($fieldName)));
				File::cleanTemp(array_keys($currentData));
			}
			if ($previousValue = $this->getPreviousValue($fieldName)) {
				$previousData = Json::decode($previousValue);
				foreach ($previousData as $item) {
					if (!isset($currentData[$item['key']])) {
						File::cleanTemp($item['key']);
						File::loadFromInfo(['path' => $item['path']])->delete();
					}
				}
			}
		}
	}

	/**
	 * Get pervious value by field.
	 *
	 * @param string $fieldName
	 *
	 * @return mixed
	 */
	public function getPreviousValue(string $fieldName = '')
	{
		return $fieldName ? ($this->changes[$fieldName] ?? null) : $this->changes;
	}

	/**
	 * Function to delete the current record model.
	 *
	 * @return int
	 */
	public function delete(): int
	{
		$return = Db::getInstance('admin')->createCommand()
			->delete(KSeF::SETTINGS_TABLE_NAME, ['id' => $this->getId()])
			->execute();
		$this->clearCache();
		return $return;
	}

	/** {@inheritdoc} */
	public function getRecordLinks(): array
	{
		$recordLinks = [];

		$recordLinks[] = [
			'linktype' => 'LISTVIEWRECORD',
			'linklabel' => 'BTN_RECORD_EDIT',
			'linkdata' => ['url' => $this->getModule()->getEditRecordUrl((int) $this->getId())],
			'linkicon' => 'yfi yfi-full-editing-view',
			'linkclass' => 'btn btn-primary btn-sm js-edit-record-modal',
		];

		$recordLinks[] = [
			'linktype' => 'LISTVIEWRECORD',
			'linklabel' => 'LBL_CHECK_CONNECTION',
			'linkurl' => 'javascript:void(0);',
			'linkdata' => ['record' => $this->getId()],
			'linkicon' => 'mdi mdi-reload',
			'linkclass' => 'btn btn-sm btn-outline-primary js-check-connection',
		];

		$recordLinks[] = [
			'linktype' => 'LISTVIEWRECORD',
			'linklabel' => 'LBL_DELETE_RECORD',
			'linkurl' => "javascript:Settings_Vtiger_List_Js.deleteById('{$this->getId()}')",
			'linkicon' => 'fas fa-trash-alt',
			'linkclass' => 'btn btn-sm btn-outline-primary',
		];

		$links = [];
		foreach ($recordLinks as $recordLink) {
			$links[] = Vtiger_Link_Model::getInstanceFromValues($recordLink);
		}
		return $links;
	}

	public function validate()
	{
		$response = [];
		if ($this->isDuplicate()) {
			$response[] = [
				'result' => false,
				'message' => Language::translate('ERR_ACTIVE_SETTINGS_EXISTS_FOR_COMPANY', $this->getModule()->getName(true))
			];
		} elseif (!$this->isConnection()) {
			$response[] = [
				'result' => false,
				'hoverField' => 'passphrase',
				'message' => Language::translate('LBL_INVALID_CONNECTION_DATA', $this->getModule()->getName(true))
			];
		}

		return $response;
	}

	public function isConnection(): bool
	{
		try {
			return $this->getConnection()->isConnected();
		} catch (Throwable $e) {
			Log::error($e->__toString(), 'KSeF');
			return false;
		}
	}

	/**
	 * Validate that only one active KSeF settings record exists per company.
	 *
	 * @return bool
	 */
	public function isDuplicate(): bool
	{
		$query = (new Query())
			->from(KSeF::SETTINGS_TABLE_NAME)
			->where(['company' => $this->get('company')]);

		if ($this->getId()) {
			$query->andWhere(['<>', 'id', $this->getId()]);
		}

		return $query->exists();
	}

	public function getConnection(bool $reload = false): ConnectorService
	{
		if (null === $this->connectorService || $reload) {
			$setting = Setting::fromData($this->getData());
			$this->connectorService = (new ConnectorService())->createConnection($setting, EncryptionKeyFactory::makeRandom());
		}

		return $this->connectorService;
	}
}
