<?php
/* +***********************************************************************************
 * The contents of this file are subject to the vtiger CRM Public License Version 1.0
 * ("License"); You may not use this file except in compliance with the License
 * The Original Code is:  vtiger CRM Open Source
 * The Initial Developer of the Original Code is vtiger.
 * Portions created by vtiger are Copyright (C) vtiger.
 * All Rights Reserved.
 * Contributor(s): YetiForce S.A.
 * *********************************************************************************** */

use App\Exceptions\NoPermitted;
use App\Exceptions\NoPermittedToRecord;
use App\Privilege;
use App\Purifier;
use App\Request;

class Vtiger_MassSave_Action extends Vtiger_Mass_Action
{
	/**
	 * Function to check permission.
	 *
	 * @param Request $request
	 *
	 * @throws NoPermitted
	 */
	public function checkPermission(Request $request)
	{
		if ($request->isEmpty('relatedModule', true)) {
			$moduleName = $request->getModule();
		} else {
			$moduleName = $request->getByType('relatedModule', Purifier::ALNUM);
			if (!Privilege::isPermitted($request->getModule(), 'DetailView', $request->getInteger('record'))) {
				throw new NoPermittedToRecord('ERR_NO_PERMISSIONS_FOR_THE_RECORD', 403);
			}
		}
		$userPriviligesModel = Users_Privileges_Model::getCurrentUserPrivilegesModel();
		if (!$userPriviligesModel->hasModuleActionPermission($moduleName, 'MassEdit')) {
			throw new NoPermitted('LBL_PERMISSION_DENIED', 406);
		}
	}

	public function process(Request $request)
	{
		$recordModels = $this->getRecordModelsFromRequest($request);
		$allRecordSave = true;
		foreach ($recordModels as $recordModel) {
			if (false !== $recordModel) {
				$recordModel->save();
				unset($recordModel);
			} else {
				$allRecordSave = false;
			}
		}
		$response = new Vtiger_Response();
		$response->setResult($allRecordSave);
		$response->emit();
	}

	/**
	 * Function to get the record model based on the request parameters.
	 *
	 * @param Request $request
	 *
	 * @return Vtiger_Record_Model[] - List of Vtiger_Record_Model instances
	 */
	public function getRecordModelsFromRequest(Request $request)
	{
		if ($request->isEmpty('relatedModule', true)) {
			$moduleName = $request->getModule();
			$recordIds = Vtiger_Mass_Action::getRecordsListFromRequest($request);
		} else {
			$moduleName = $request->getByType('relatedModule', Purifier::ALNUM);
			$recordIds = Vtiger_RelationAjax_Action::getRecordsListFromRequest($request);
		}
		$moduleModel = Vtiger_Module_Model::getInstance($moduleName);

		$recordModels = [];
		$fieldModelList = $moduleModel->getFields();
		foreach ($recordIds as $recordId) {
			$recordModel = Vtiger_Record_Model::getInstanceById($recordId, $moduleModel);
			if (!$recordModel->isEditable()) {
				$recordModels[$recordId] = false;
				continue;
			}
			foreach ($fieldModelList as $fieldName => $fieldModel) {
				if ($fieldModel->isWritable() && $request->has($fieldName)) {
					$fieldUiTypeModel = $fieldModel->getUITypeModel();
					if (!method_exists($fieldUiTypeModel, 'setValueFromMassEdit') || !$fieldUiTypeModel->setValueFromMassEdit($request, $recordModel)) {
						$fieldUiTypeModel->setValueFromRequest($request, $recordModel);
					}
				}
				$recordModels[$recordId] = $recordModel;
			}
		}
		return $recordModels;
	}
}
