<?php

use App\Exceptions\Security;
use App\Fields\Currency;
use App\Fields\Double;
use App\Json;
use App\Purifier;
use App\TextUtils;
use App\Validator;

/**
 * Inventory Tax Field Class.
 *
 * @package   InventoryField
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
class Vtiger_Tax_InventoryField extends Vtiger_Basic_InventoryField
{
	/** @var array List of shared fields */
	public $shared = ['taxparam' => 'tax_percent'];
	protected $type = 'Tax';
	protected $defaultLabel = 'LBL_TAX';
	protected $defaultValue = 0;
	protected $columnName = 'tax';
	protected $dbType = 'decimal(28,8) DEFAULT 0';
	protected $customColumn = [
		'taxparam' => 'string'
	];
	protected $summationValue = true;
	protected $maximumLength = '99999999999999999999';
	protected $customMaximumLength = [
		'taxparam' => 255
	];
	protected $purifyType = Purifier::NUMBER;
	protected $customPurifyType = [
		'taxparam' => Purifier::TEXT
	];
	/** {@inheritdoc} */
	protected $params = ['summary_enabled'];
	protected bool $calculated = true;

	/** {@inheritdoc} */
	public function getDisplayValue($value, array $rowData = [], bool $rawText = false)
	{
		$value = Double::formatToDisplay($value);
		if (isset($rowData['currency']) && $currencySymbol = Currency::getById($rowData['currency'])['currency_symbol'] ?? '') {
			$value = CurrencyField::appendCurrencySymbol($value, $currencySymbol);
		}

		return $value;
	}

	/** {@inheritdoc} */
	public function getEditValue(array $itemData, string $column = '')
	{
		$value = parent::getEditValue($itemData, $column);
		if (!$column || $column === $this->getColumnName()) {
			$value = Double::formatToDisplay($value);
		}

		return $value;
	}

	public function getClassName($data)
	{
		if (!empty($data) && 0 == $data[0]['taxmode']) {
			return 'hide';
		}
		return '';
	}

	/** {@inheritdoc} */
	public function getDBValue($value, ?string $name = '')
	{
		if ($name !== $this->getColumnName()) {
			$valid = $value ? Json::decode($value) : [];
			if (isset($valid['individualTax'])) {
				$valid['individualTax'] ??= 0;
				$valid['globalTax'] ??= 0;
				$value = Json::encode($valid);
			}
		} else {
			$value = Double::formatToDb($value);
		}
		return $value;
	}

	/** {@inheritdoc} */
	public function validate($value, string $columnName, bool $isUserFormat, $originalValue = null)
	{
		if ($columnName === $this->getColumnName()) {
			if ($isUserFormat) {
				$value = $this->getDBValue($value, $columnName);
				if (null !== $originalValue) {
					$originalValue = $this->getDBValue($originalValue, $columnName);
				}
			}
			if (!is_numeric($value)) {
				throw new Security("ERR_ILLEGAL_FIELD_VALUE||$columnName||$value", 406);
			}
			if ($this->maximumLength < $value || -$this->maximumLength > $value) {
				throw new Security("ERR_VALUE_IS_TOO_LONG||$columnName||$value", 406);
			}
			if (null !== $originalValue && !Validator::floatIsEqualUserCurrencyDecimals($value, $originalValue)) {
				throw new Security('ERR_ILLEGAL_FIELD_VALUE||' . ($columnName ?? $this->getColumnName()) . "||{$this->getModuleName()}||$value($originalValue)", 406);
			}
		} else {
			if (TextUtils::getTextLength($value) > $this->customMaximumLength[$columnName]) {
				$module = $this->getModuleName();
				throw new Security("ERR_VALUE_IS_TOO_LONG||$columnName||$module||$value", 406);
			}
		}
	}

	/**
	 * Get configuration parameters for taxes.
	 *
	 * @param string $taxParam String parameters json encode
	 * @param float  $net
	 * @param array  $return
	 *
	 * @return array
	 */
	public function getTaxParam(string $taxParam, float $net, array $return = []): array
	{
		$taxParam = json_decode($taxParam, true);
		if (empty($taxParam)) {
			return $return;
		}
		if (\is_string($taxParam['aggregationType'])) {
			$taxParam['aggregationType'] = [$taxParam['aggregationType']];
		}
		if (isset($taxParam['aggregationType'])) {
			foreach ($taxParam['aggregationType'] as $aggregationType) {
				$percent = (string) ($taxParam[$aggregationType . 'Tax'] ?? 0);
				if (!isset($return[$percent])) {
					$return[$percent] = 0;
				}
				$return[$percent] += $net * ($percent / 100);
			}
			ksort($return);
		}

		return $return;
	}

	/** {@inheritdoc} */
	public function getValueForSave(array $item, bool $userFormat = false, ?string $column = null)
	{
		if ($column === $this->getColumnName() || null === $column) {
			$value = 0.0;
			if (!Json::isEmpty($item['taxparam'] ?? '') && ($taxesConfig = Vtiger_Inventory_Model::getTaxesConfig())) {
				$taxParam = Json::decode($item['taxparam']);
				$netPrice = static::getInstance($this->getModuleName(), 'NetPrice')->getValueForSave($item, $userFormat);
				$value = $this->getTaxValue($taxParam, $netPrice, (int) $taxesConfig['aggregation']);
			}
		} else {
			$value = $item[$column] ?? $this->getDefaultValue($column);
			$value = $userFormat ? $this->getDBValue($value) : $value;
		}
		return $value;
	}

	/**
	 * Calculate the tax value.
	 *
	 * @param array  $taxParam
	 * @param float  $netPrice
	 * @param string $mode     0-can not be combined, 1-summary, 2-cascade
	 *
	 * @return float
	 */
	public function getTaxValue(array $taxParam, float $netPrice, int $mode): float
	{
		$value = 0.0;
		if ($taxParam) {
			$types = $taxParam['aggregationType'];
			if (!\is_array($types)) {
				$types = [$types];
			}
			foreach ($types as $type) {
				$taxValue = $netPrice * $taxParam["{$type}Tax"] / 100.00;
				$value += $taxValue;
				if (2 === $mode) {
					$netPrice += $taxValue;
				}
			}
		}
		return $this->roundDecimal($value);
	}

	/** {@inheritdoc} */
	public function compare($value, $prevValue, string $column): bool
	{
		return $column === $this->getColumnName() ? Validator::floatIsEqual((float) $value, (float) $prevValue, 8) : parent::compare($value, $prevValue, $column);
	}

	/**
	 * Returns tax summary for each tax rate.
	 *
	 * @param Vtiger_Record_Model $model
	 *
	 * @return array
	 */
	public function getTaxSummary(Vtiger_Record_Model $model): array
	{
		$inventory = $model->getInventoryData();
		$groups = [];
		$total = 0;

		foreach ($inventory as $data) {
			if (Json::isEmpty($data['taxparam'] ?? '') || !($taxesConfig = Vtiger_Inventory_Model::getTaxesConfig())) {
				continue;
			}

			$taxParam = Json::decode($data['taxparam']);
			$netPrice = static::getInstance($this->getModuleName(), 'NetPrice')->getValueForSave($data);

			$types = $taxParam['aggregationType'];
			if (!\is_array($types)) {
				$types = [$types];
			}

			foreach ($types as $type) {
				$taxPercent = $taxParam["{$type}Tax"];
				$taxValue = $netPrice * $taxPercent / 100.00;
				$taxKey = static::getInstance($this->getModuleName(), 'TaxPercent')->getDisplayValue($taxPercent, $data);
				$taxKey .= (is_numeric($taxKey) ? '%' : '');
				$groups[$taxKey] = ($groups[$taxKey] ?? 0) + $taxValue;
				$total += $taxValue;
				if (2 === (int) $taxesConfig['aggregation']) {
					$netPrice += $taxValue;
				}
			}
		}

		foreach ($groups as $key => $row) {
			$groups[$key] = $this->roundDecimal($row);
		}

		ksort($groups, SORT_NUMERIC);

		return [
			'groups' => $groups,
			'total' => $this->roundDecimal($total),
		];
	}

	/** {@inheritDoc} */
	public function getSummaryPicklist(): array
	{
		return [
			self::SUMMARY_ON => 'PLL_INV_SHOW_SUMMARY_FIELD',
			self::SUMMARY_BLOCK => 'PLL_INV_SHOW_SUMMARY_BLOCK',
			self::SUMMARY_BY_GROUP => 'PLL_INV_SHOW_SUMMARY_BY_GROUP',
		];
	}
}
