<?php
/**
 * CellMerger class
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Leszek Koziatek <l.koziatek@yetiforce.com>
 */
declare(strict_types=1);

namespace App\DocumentBuilder\Services;

use PhpOffice\PhpSpreadsheet\Cell\Cell;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

/** CellMerger class */
final class CellMerger
{
	public function __construct(
		readonly private Worksheet $worksheet
	) {
	}

	/**
	 * Merge similar cells in columns
	 *
	 * @param array $groupedBy array with column indexes (eg. ['A','B'] where merge cell should be applied)
	 *
	 * @throws \PhpOffice\PhpSpreadsheet\Exception
	 */
	public function mergeColumns(array $groupedBy = []): Worksheet
	{
		$mergeCells = [];

		if (Coordinate::columnIndexFromString($this->worksheet->getHighestColumn()) === 1) {
			return $this->worksheet;
		}

		foreach ($this->worksheet->getColumnIterator() as $column) {
			$lastMatchedValue = null;
			$lastMatchedCoordinate = null;
			$this->worksheet->getColumnDimension($column->getColumnIndex())->setAutoSize(true);
			foreach ($column->getCellIterator() as $cell) {
				if ($groupedBy && !in_array($cell->getColumn(), $groupedBy)) {
					continue;
				}
				$cellValue = $cell->getValue();
				if (
					null === $cellValue
					|| $lastMatchedValue !== $cellValue
					|| false === $this->hasSameGroup($cell, $lastMatchedCoordinate)
				) {
					$lastMatchedValue = $cellValue;
					$lastMatchedCoordinate = $cell->getCoordinate();
					continue;
				}

				$cellCoordinate = $cell->getCoordinate();
				$mergeCells[$lastMatchedCoordinate] = $cellCoordinate;
			}
		}

		foreach ($mergeCells as $startCoordinate => $endCoordinate) {
			$this->worksheet->mergeCells(
				sprintf('%s:%s', $startCoordinate, $endCoordinate),
				Worksheet::MERGE_CELL_CONTENT_HIDE,
			);
		}

		return $this->worksheet;
	}

	/**
	 * Check if cell has the same parent as last checked cell
	 *
	 * @throws \PhpOffice\PhpSpreadsheet\Exception
	 */
	private function hasSameGroup(Cell $cell, ?string $lastMatchedCoordinate): bool
	{
		$initialCellColumnNumber = Coordinate::columnIndexFromString($cell->getColumn());
		$initialCellRowNumber = $cell->getRow();

		if (null === $lastMatchedCoordinate) {
			return true;
		}

		$currentCellPath = [];
		for ($currentColumn = $initialCellColumnNumber - 1; $currentColumn > 0; $currentColumn--) {
			$currentCellPath[] = $this->worksheet->getCell([$currentColumn, $initialCellRowNumber])->getValue();
		}

		$lastMatchedCell = $this->worksheet->getCell($lastMatchedCoordinate);
		$initialLastMatchedCellColumn = Coordinate::columnIndexFromString($lastMatchedCell->getColumn());
		$initialLastMatchedCellRow = $lastMatchedCell->getRow();

		$lastMatchedCellPath = [];
		for ($currentColumn = $initialLastMatchedCellColumn - 1; $currentColumn > 0; $currentColumn--) {
			$lastMatchedCellPath[] = $this->worksheet->getCell([$currentColumn, $initialLastMatchedCellRow])->getValue();
		}

		$this->worksheet->getCell([$initialCellColumnNumber, $initialCellRowNumber]);

		return $currentCellPath === $lastMatchedCellPath;
	}
}
