<?php

/**
 * Service provide unified alias for Query Generator.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Leszek Koziatek <l.koziatek@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Report\Provider;

use App\FieldCoordinatorTransformer\QueryGeneratorFieldTransformer;
use App\Report\Enums\FunctionType;

/** AliasProvider class */
final class AliasProvider
{
	/** Provide alias in format field_name:module_name or filed_name:module_name:INVENTORY for inventory fields */
	public static function provide(string $fieldCoordinate, ?string $function = null): string
	{
		$parts = QueryGeneratorFieldTransformer::extract($fieldCoordinate);

		if (null === $parts['moduleName']) {
			return $parts['fieldName'];
		}

		$format = QueryGeneratorFieldTransformer::isInventory($fieldCoordinate)
		&& QueryGeneratorFieldTransformer::INVENTORY_NAME !== $parts['referenceField']
			? '%s:%s:%s:%s'
			: '%s:%s:%s';

		if (null !== $function) {
			$format = \sprintf('%s_%s', $format, $function);
		}

		return \sprintf(
			$format,
			$parts['fieldName'],
			$parts['moduleName'],
			$parts['referenceField'],
			QueryGeneratorFieldTransformer::INVENTORY_NAME,
		);
	}

	/** Provide alias for formula expression eg. sum(net) */
	public static function provideFormulaAlias(FunctionType $function, array $fields): string
	{
		$aliasesFields = [];
		foreach ($fields as $field) {
			$aliasesFields[] = self::provide($field);
		}

		return \sprintf('%s(%s)', $function->value, implode(',', $aliasesFields));
	}
}
