<?php

/**
 * Common repository for db connected queries.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Łukasz Krawczyk <l.krawczyk@yetiforce.com>
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Report\Repository\Template;

use App\Db\Query;
use App\Report\Builder\TemplateBuilder;
use App\Report\Model\Template;
use App\Report\Repository\TemplateRepository;

/** YiiTemplateRepository class */
final class YiiTemplateRepository implements TemplateRepository
{
	public function __construct(
		private readonly TemplateBuilder $templateBuilder,
	) {}

	/** {@inheritDoc} */
	public function find(int $id): ?Template
	{
		$result = (new Query())
			->select(['a.name', 'a.query_id', 'modulename' => 'c.name'])
			->from(['a' => 'u_yf_report_template'])
			->innerJoin(['b' => 'u_yf_report_template_query'], 'a.query_id = b.id')
			->innerJoin(['c' => 'vtiger_tab'], 'c.tabid = b.base_module_id')
			->where(['a.id' => $id])->one();

		if ($result) {
			$queryId = $result['query_id'];
			$moduleName = $result['modulename'];
			$name = $result['name'];

			$expressions = $this->findExpressionsByQueryId($queryId);
			$filters = $this->findFiltersByQueryId($queryId);

			$query = $this->templateBuilder->buildQuery($moduleName, $queryId);
			return $this->templateBuilder->build($query, $expressions, $filters, $name, $id);
		}

		return null;
	}

	public function findExpressionsByQueryId(int $queryId)
	{
		$expressions = [];

		$dataReader = (new Query())
			->from('u_yf_report_template_query_expression')
			->where(['query_id' => $queryId])
			->addOrderBy(['sort' => SORT_ASC])
			->createCommand()
			->query();
		while ($row = $dataReader->read()) {
			$expressions[$row['id']]['field'] = $row['field'];
			$expressions[$row['id']]['function'] = $row['function'];
			$expressions[$row['id']]['group_by'] = $row['group_by'];
			$expressions[$row['id']]['order_by'] = $row['order_by'];
			$expressions[$row['id']]['sort'] = (int) $row['sort'];
			$expressions[$row['id']]['formula_field'] = $row['formula_field'];
		}

		return $expressions;
	}

	public function findFiltersByQueryId(int $queryId): array
	{
		$dataReader = (new Query())->select([
			'b.filter_group_id',
			'b.field',
			'b.operator',
			'b.value',
			'condition_index' => 'b.id',
			'a.id',
			'a.condition',
			'a.parent_id',
			'group_index' => 'a.id',
		])->from('u_yf_report_template_query_filter_group a')
			->leftJoin('u_yf_report_template_query_filter b', 'a.id = b.filter_group_id')
			->where(['a.query_id' => $queryId])
			->orderBy(['a.parent_id' => SORT_ASC])
			->createCommand()->query();
		$referenceGroup = $referenceParent = $conditions = [];
		while ($condition = $dataReader->read()) {
			if ($condition['filter_group_id']) {
				$isEmptyCondition = false;
			} else {
				$condition['filter_group_id'] = $condition['id'];
				$isEmptyCondition = true;
			}
			$fieldCond = [
				'field' => $condition['field'],
				'operator' => $condition['operator'],
				'value' => $condition['value'],
			];

			if (isset($referenceParent[$condition['parent_id']], $referenceGroup[$condition['filter_group_id']])) {
				$referenceParent[$condition['parent_id']][$condition['condition_index']] = $fieldCond;
			} elseif (isset($referenceGroup[$condition['parent_id']])) {
				if ($isEmptyCondition) {
					$referenceGroup[$condition['parent_id']][$condition['group_index']] = [
						'condition' => $condition['condition'],
						'rules' => [],
					];
				} else {
					$referenceGroup[$condition['parent_id']][$condition['group_index']] = [
						'condition' => $condition['condition'],
						'rules' => [
							$condition['condition_index'] => $fieldCond,
						],
					];
				}
				$referenceParent[$condition['parent_id']] = &$referenceGroup[$condition['parent_id']][$condition['group_index']]['rules'];
				$referenceGroup[$condition['filter_group_id']] = &$referenceGroup[$condition['parent_id']][$condition['group_index']]['rules'];
			} else {
				if ($isEmptyCondition) {
					$conditions = [
						'condition' => $condition['condition'],
						'rules' => [],
					];
				} else {
					$conditions = [
						'condition' => $condition['condition'],
						'rules' => [$fieldCond],
					];
				}
				$referenceParent[$condition['parent_id']] = &$conditions['rules'];
				$referenceGroup[$condition['filter_group_id']] = &$conditions['rules'];
			}
		}

		return $this->sortConditions($conditions);
	}

	/**
	 * Merge expression field data with YetiForce field definition.
	 *
	 * @param int    $id
	 * @param string $fieldName
	 * @param string $moduleName
	 *
	 * @return array{field: string, label: string, moduleName: string}|null
	 */
	public function getFieldDefinition(int $id, string $fieldName, string $moduleName): ?array
	{
		$field = (new Query())
			->select(['uyrtqe.field', 'vf.fieldlabel as label', 'vt.name as moduleName'])
			->from('u_yf_report_template ufrt')
			->innerJoin('u_yf_report_template_query_expression uyrtqe', 'ufrt.query_id = uyrtqe.query_id')
			->innerJoin('vtiger_field vf', 'vf.fieldname = uyrtqe.field')
			->innerJoin('vtiger_tab vt', 'vt.tabid = vf.tabid')
			->andWhere(['ufrt.id' => $id])
			->andWhere(['uyrtqe.field' => $fieldName])
			->andWhere(['vt.name' => $moduleName])
			->one();

		return false !== $field ? $field : null;
	}

	private function sortConditions(array $arrayToSort): array
	{
		if (isset($arrayToSort['rules'])) {
			ksort($arrayToSort['rules']);
			foreach ($arrayToSort['rules'] as $rule) {
				if (isset($rule['condition'])) {
					self::sortConditions($rule);
				}
			}
		}
		return $arrayToSort;
	}
}
