<?php
/**
 * YetiForce register file.
 * Modifying this file or functions that affect the footer appearance will violate the license terms!!!
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

namespace App\YetiForce;

use App\Exceptions\AppException;
use App\Json;
use App\Language;
use App\Log;
use App\RequestHttp;
use App\RequestUtil;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ServerException;

/**
 * YetiForce register class.
 */
final class ApiClient
{
	/** @var ?string Last error. */
	public ?string $error = null;

	/** @var bool Response result */
	private bool $success;

	/** @var int|null Response code */
	private ?int $responseCode = 0;

	/** @var string Response body */
	private string $responseBody;

	/** @var int Total timeout of the request in seconds. */
	private int $timeout = 20;

	/** @var int The number of seconds to wait while trying to connect to a server. */
	private int $connectTimeout = 10;

	/**
	 * Send registration data.
	 *
	 * @param string $url
	 * @param string $method
	 * @param array $option
	 *
	 * @return bool
	 * @throws AppException
	 */
	public function send(string $url, string $method, array $option = []): bool
	{
		$this->error = null;
		$this->success = false;
		$this->basicValidations($url);
		if ($this->error) {
			return $this->success;
		}

		try {
			Log::beginProfile($method . '|' . __METHOD__ . "|$url", __NAMESPACE__);
			$response = (new Client($this->getRequestOptions()))->request($method, $url, $option);
			Log::endProfile($method . '|' . __METHOD__ . "|$url", __NAMESPACE__);

			$this->responseCode = $response->getStatusCode();
			$this->responseBody = $response->getBody()->getContents();
			$this->success = true;
		} catch (ClientException $e) {
			$this->responseCode = $e->getResponse()->getStatusCode();
			$this->error = $e->getResponse()->getBody()->getContents();
			if (Json::isJson($this->error) && ($error = Json::decode($this->error)['errors'] ?? null)) {
				$this->error = is_array($error) ? implode(' | ', $error) : $error;
			}
			Log::error($e->getMessage(), __METHOD__);
		} catch (ServerException $e) {
			$this->responseCode = $e->getResponse()->getStatusCode();
			$this->error = $this->responseCode . ' Internal Server Error';
			Log::error($e->getMessage(), __METHOD__);
		} catch (\Throwable $e) {
			$this->error = Language::translate("LBL_ERROR");
			Log::error($e->getMessage(), __METHOD__);
		}

		return $this->success;
	}

	/**
	 * Get response status code.
	 *
	 * @return ?int
	 */
	public function getStatusCode(): ?int
	{
		return $this->responseCode;
	}

	/**
	 * Get response content.
	 *
	 * @return string
	 */
	public function getResponseBody(): string
	{
		return $this->responseBody;
	}

	/**
	 * Get last error.
	 *
	 * @return string
	 */
	public function getError(): string
	{
		return $this->error ?? '';
	}

	/**
	 * Get request options.
	 *
	 * @return array
	 * @throws \ReflectionException
	 */
	public function getRequestOptions(): array
	{
		$headers = [
			'x-crm-id' => \App\Config::main('application_unique_key'),
			'x-app-id' => Register::getInstanceKey(),
			'accept-language' => Language::getLanguage() ?: 'en'
		];
		if ($key = (new Config())->getToken()) {
			$headers['x-api-key'] = $key;
		}

		$options = RequestHttp::getOptions();

		return array_merge($options, [
			'headers' => array_merge($options['headers'] ?? [], $headers),
			'timeout' => $this->timeout,
			'connect_timeout' => $this->connectTimeout
		]);
	}

	/**
	 * Basic validations.
	 *
	 * @param string $url
	 *
	 * @return void
	 */
	private function basicValidations(string $url): void
	{
		$hostName = parse_url($url, PHP_URL_HOST);
		if (!RequestUtil::isNetConnection() || $hostName === gethostbyname($hostName)) {
			Log::warning('ERR_NO_INTERNET_CONNECTION', __METHOD__);
			$this->error = 'ERR_NO_INTERNET_CONNECTION';
		} elseif (!$this->isWritable()) {
			Log::warning('ERR_REGISTER_FILES_PERMISSIONS||app_data', __METHOD__);
			$this->error = 'ERR_REGISTER_FILES_PERMISSIONS||app_data';
		}
	}

	/**
	 * Check write permissions for the registry file.
	 *
	 * @return bool
	 */
	private function isWritable(): bool
	{
		$path = Register::REGISTRATION_FILE;
		return (file_exists($path) && is_writable($path)) || (!file_exists($path) && is_writable(dirname($path)));
	}
}
