<?php
/**
 * UserAuth record model file.
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 * @author Antoni Kiszka <a.kiszka@yetiforce.com>
 */

/**
 * UserAuth record model file.
 */
class Settings_UserAuth_Record_Model extends Settings_Vtiger_Record_Model
{
	/** @var array Record changes */
	protected $changes = [];
	/** @var \App\Authenticator\AbstractProvider Provider object */
	private $provider;

	/**
	 * Edit fields.
	 *
	 * @var string[]
	 */
	private $editFields = ['name', 'isactive'];

	/**
	 * Function to get Id of this record instance.
	 *
	 * @return int Id
	 */
	public function getId()
	{
		return $this->get('id');
	}

	/**
	 * Function to get Name of this record instance.
	 *
	 * @return string Name
	 */
	public function getName()
	{
		return '';
	}

	/**
	 * Function to get module of this record instance.
	 *
	 * @return Settings_UserAuth_Module_Model $moduleModel
	 */
	public function getModule()
	{
		return $this->module;
	}

	/**
	 * Function to set module instance to this record instance.
	 *
	 * @param Settings_UserAuth_Module_Model $moduleModel
	 *
	 * @return Settings_UserAuth_Module_Model this record
	 */
	public function setModule($moduleModel)
	{
		$this->module = $moduleModel;
		return $this;
	}

	/**
	 * Function to get Edit view url.
	 *
	 * @return string Url
	 */
	public function getEditViewUrl()
	{
		return \App\UserAuth::getProviderByName($this->get('providertype'))->getEditViewUrl() . '&record=' . $this->getId();
	}

	/** {@inheritdoc} */
	public function getRecordLinks(): array
	{
		$links = [];
		$recordLinks = [
			[
				'linktype' => 'LISTVIEWRECORD',
				'linklabel' => 'LBL_EDIT_RECORD',
				'linkurl' => $this->getEditViewUrl() . '&record=' . $this->getId(),
				'linkicon' => 'yfi yfi-full-editing-view',
				'linkclass' => 'btn btn-sm btn-primary',
				'modalView' => true,
			],
			[
				'linktype' => 'LISTVIEWRECORD',
				'linklabel' => 'LBL_DELETE_RECORD',
				'linkurl' => 'javascript:Settings_Vtiger_List_Js.deleteById(' . $this->getId() . ');',
				'linkicon' => 'fas fa-trash-alt',
				'linkclass' => 'btn btn-sm btn-outline-primary',
			],
		];
		foreach ($recordLinks as $recordLink) {
			$links[] = Vtiger_Link_Model::getInstanceFromValues($recordLink);
		}
		return $links;
	}

	/**
	 * Auth provider.
	 *
	 * @return App\Authenticator\AbstractProvider
	 */
	public function getProvider(): App\Authenticator\AbstractProvider
	{
		if (null === $this->provider) {
			$this->provider = \App\UserAuth::getProviderByName($this->get('providertype'));
		}

		return $this->provider;
	}

	/**
	 * Function to getDisplay value of every field.
	 *
	 * @param string $name field name
	 *
	 * @return mixed
	 */
	public function getDisplayValue(string $name)
	{
		switch ($name) {
			case 'providertype':
				$displayValue = \App\UserAuth::getProviderByName($this->get('providertype'))->getLabel();
				break;
			case 'isactive':
				$moduleName = $this->getModule()->getName(true);
				$displayValue = empty($this->get($name)) ? \App\Language::translate('FL_INACTIVE', $moduleName) : \App\Language::translate('FL_ACTIVE', $moduleName);
				break;
			default:
				$displayValue = App\Purifier::encodeHtml($this->get($name));
		}
		return $displayValue;
	}

	/**
	 * Function to save.
	 */
	public function save()
	{
		$result = false;
		$db = App\Db::getInstance('admin');
		$transaction = $db->beginTransaction();
		try {
			$this->saveToDb();
			$transaction->commit();
			$result = true;
		} catch (\Throwable $ex) {
			$transaction->rollBack();
			\App\Log::error($ex->__toString());
			throw $ex;
		}
		$this->clearCache();
		return $result;
	}

	/**
	 * Save data to the database.
	 */
	public function saveToDb()
	{
		$db = \App\Db::getInstance('admin');
		$fields = array_flip(['name', 'providertype', 'isactive', 'config']);
		$tablesData = $this->getId() ? array_intersect_key($this->getData(), $this->changes, $fields) : array_intersect_key($this->getData(), $fields);
		if ($tablesData) {
			$baseTable = $this->getModule()->baseTable;
			$baseTableIndex = $this->getModule()->baseIndex;
			if ($this->getId()) {
				$db->createCommand()->update($baseTable, $tablesData, [$baseTableIndex => (int) $this->getId()])->execute();
			} else {
				$db->createCommand()->insert($baseTable, $tablesData)->execute();
				$this->set('id', $db->getLastInsertID("{$baseTable}_{$baseTableIndex}_seq"));
			}
		}
	}

	/**
	 * Get pervious value by field.
	 *
	 * @param string $fieldName
	 *
	 * @return mixed
	 */
	public function getPreviousValue(string $fieldName = '')
	{
		return $fieldName ? ($this->changes[$fieldName] ?? null) : $this->changes;
	}

	/**
	 * Sets data from request.
	 *
	 * @param App\Request $request
	 */
	public function setDataFromRequest(App\Request $request)
	{
		$config = $this->getConfig();
		foreach ($this->getEditFields() as $fieldName => $fieldModel) {
			if ($request->has($fieldName)) {
				$value = $request->isEmpty($fieldName) && !$fieldModel->isMandatory() ? '' : $request->getByType($fieldName, $fieldModel->get('purifyType'));
				$fieldModel->getUITypeModel()->validate($value, true);
				$value = $fieldModel->getUITypeModel()->getDBValue($value);

				if (\in_array($fieldName, ['id', 'name', 'providertype', 'isactive'])) {
					$this->set($fieldName, $value);
				} else {
					$config[$fieldName] = $value;
					$this->set('config', \App\Json::encode($config));
				}
			}
		}
	}

	/**
	 * Clear cache.
	 */
	public function clearCache()
	{
		\App\Cache::delete('UserAuthConfig', $this->get('providertype'));
	}

	/**
	 * Function to set the value for a given key.
	 *
	 * @param string $key
	 * @param mixed  $value
	 */
	public function set($key, $value)
	{
		if ($this->getId() && !\in_array($key, ['id']) && (\array_key_exists($key, $this->value) && $this->value[$key] != $value) && !\array_key_exists($key, $this->changes)) {
			$this->changes[$key] = $this->get($key);
		}
		return parent::set($key, $value);
	}

	/**
	 * Data anonymization.
	 *
	 * @param array $data
	 *
	 * @return array
	 */
	public function anonymize(array $data): array
	{
		$provider = $this->getProvider();
		foreach ($data as $key => &$value) {
			if ('config' === $key && !empty($value)) {
				$decoded = \App\Json::decode($value);
				foreach ($provider->getEncryptedFields() as $field) {
					if (\array_key_exists($field, $decoded)) {
						$decoded[$field] = '****';
					}
				}
				$value = \App\Json::encode($decoded);
			}
		}

		return $data;
	}

	/**
	 * Function to get the instance, given id.
	 *
	 * @param int $id
	 *
	 * @return \self
	 */
	public static function getInstanceById($id)
	{
		$instance = self::getCleanInstance();
		$data = (new App\Db\Query())
			->from($instance->getModule()->getBaseTable())
			->where([$instance->getModule()->getBaseIndex() => $id])
			->one(\App\Db::getInstance('admin'));
		$instance->setData($data);

		return $instance;
	}

	/**
	 * Function to get the instance, given the provider name.
	 *
	 * @param string $providerName
	 *
	 * @return \self
	 */
	public static function getInstanceByProvider($providerName)
	{
		$instance = self::getCleanInstance();
		$data = (new App\Db\Query())
			->from($instance->getModule()->getBaseTable())
			->where([$instance->getModule()->getSecondaryIndex() => $providerName])
			->one(\App\Db::getInstance('admin'));
		$instance->setData($data);

		return $instance;
	}

	/**
	 * Function to get clean record instance by using moduleName.
	 *
	 * @param string $qualifiedModuleName
	 * @param string $provider
	 *
	 * @return \self
	 */
	public static function getCleanInstance(?string $provider = null)
	{
		$recordModel = new self();
		$moduleModel = Settings_Vtiger_Module_Model::getInstance('Settings:UserAuth');
		$recordModel->set('providertype', $provider);

		return $recordModel->setModule($moduleModel);
	}

	/**
	 * Function determines fields available in edition view.
	 *
	 * @param string $name
	 *
	 * @return \Vtiger_Field_Model
	 */
	public function getFieldInstanceByName($name)
	{
		$moduleName = $this->getModule()->getName(true);
		$params = ['uitype' => 1, 'column' => $name, 'name' => $name, 'displaytype' => 1, 'typeofdata' => 'V~M', 'presence' => 0, 'isEditableReadOnly' => false];
		switch ($name) {
			case 'providertype':
				$params['uitype'] = 16;
				$params['picklistValues'] = [];
				$params['label'] = 'FL_PROVIDER';
				$params['displaytype'] = 2;
				$params['purifyType'] = \App\Purifier::STANDARD;
				$params['fieldvalue'] = $this->getValueByField($name);
				foreach (\App\UserAuth::getProviders() as $provider) {
					$params['picklistValues'][$provider->getName()] = \App\Language::translate($provider->getLabel(), $moduleName);
				}
				break;
			case 'isactive':
				$params['uitype'] = 16;
				$params['label'] = 'FL_STATUS';
				$params['purifyType'] = \App\Purifier::INTEGER;
				$params['fieldvalue'] = $this->getValueByField($name);
				$params['picklistValues'] = [1 => \App\Language::translate('FL_ACTIVE'), 0 => \App\Language::translate('FL_INACTIVE')];
				break;
			case 'name':
				$params['uitype'] = 1;
				$params['label'] = 'FL_NAME';
				$params['purifyType'] = \App\Purifier::TEXT;
				$params['fieldvalue'] = $this->getValueByField($name);
				$params['maximumlength'] = 50;
				break;
			default:
				break;
		}

		return Settings_Vtiger_Field_Model::init($moduleName, $params);
	}

	/**
	 * Function determines fields available in edition view.
	 *
	 * @return Vtiger_Field_Model[]
	 */
	public function getEditFields()
	{
		$fields = [];
		foreach ($this->editFields as $fieldName) {
			$fields[$fieldName] = $this->getFieldInstanceByName($fieldName);
		}
		$provider = $this->getProvider();
		foreach ($provider->getEditFields() as $fieldName => $fieldModel) {
			$fieldModel->set('fieldvalue', $this->getValueByField($fieldName));
			$fields[$fieldName] = $fieldModel;
		}

		return $fields;
	}

	/**
	 * Get config.
	 *
	 * @return array
	 */
	public function getConfig(): array
	{
		return $this->get('config') ? \App\Json::decode($this->get('config')) : [];
	}

	/**
	 * Get parameter value by name.
	 *
	 * @param string $fieldName
	 *
	 * @return string
	 */
	public function getConfigValue(string $fieldName): string
	{
		return $this->getConfig()[$fieldName] ?? '';
	}

	/**
	 * Get value by name.
	 *
	 * @param string $fieldName
	 *
	 * @return mixed
	 */
	public function getValueByField(string $fieldName)
	{
		return \array_key_exists($fieldName, $this->value) ? $this->value[$fieldName] : $this->getConfigValue($fieldName);
	}

	/**
	 * Function removes record.
	 *
	 * @return bool
	 */
	public function delete()
	{
		$db = App\Db::getInstance('admin');
		$recordId = $this->getId();
		if ($recordId) {
			$table = $this->getModule()->getBaseTable();
			$index = $this->getModule()->getBaseIndex();
			$result = $db->createCommand()->delete($table, [$index => $recordId])->execute();
			$this->clearCache();
		}
		return !empty($result);
	}
}
