<?php
/* +***********************************************************************************
 * The contents of this file are subject to the vtiger CRM Public License Version 1.0
 * ("License"); You may not use this file except in compliance with the License
 * The Original Code is:  vtiger CRM Open Source
 * The Initial Developer of the Original Code is vtiger.
 * Portions created by vtiger are Copyright (C) vtiger.
 * All Rights Reserved.
 * Contributor(s): YetiForce S.A.
 * *********************************************************************************** */

use App\Conditions\QueryFields\IntegerField;
use App\Exceptions\Security;
use App\Fields\Currency;
use App\Purifier;
use App\User;

class Vtiger_Currency_UIType extends Vtiger_Base_UIType
{
	protected $edit = false;

	/** {@inheritdoc} */
	public function getDBValue($value, $recordModel = false)
	{
		return CurrencyField::convertToDBFormat($value, null, 72 === $this->getFieldModel()->get('uitype'));
	}

	/** {@inheritdoc} */
	public function validate($value, $isUserFormat = false)
	{
		if (empty($value) || isset($this->validate["$value"])) {
			return;
		}
		if ($isUserFormat) {
			$value = Currency::formatToDb($value);
		}
		if (!is_numeric($value)) {
			throw new Security('ERR_ILLEGAL_FIELD_VALUE||' . $this->getFieldModel()->getName() . '||' . $this->getFieldModel()->getModuleName() . '||' . $value, 406);
		}
		if ($maximumLength = $this->getFieldModel()->get('maximumlength')) {
			[$minimumLength, $maximumLength] = false !== strpos($maximumLength, ',') ? explode(',', $maximumLength) : [-$maximumLength, $maximumLength];
			if ((float) $minimumLength > $value || (float) $maximumLength < $value) {
				throw new Security('ERR_VALUE_IS_TOO_LONG||' . $this->getFieldModel()->getName() . '||' . $this->getFieldModel()->getModuleName() . "||{$maximumLength} < {$value} < {$minimumLength}", 406);
			}
		}
		$this->validate["$value"] = true;
	}

	/** {@inheritdoc} */
	public function getDisplayValue($value, $record = false, $recordModel = false, $rawText = false, $length = false)
	{
		if (!$value) {
			return 0;
		}
		$uiType = $this->getFieldModel()->get('uitype');
		// Some of the currency fields like Unit Price, Totoal , Sub-total - doesn't need currency conversion during save
		$value = Currency::formatToDisplay($value, null, 72 === $uiType);
		if (!$this->edit) {
			$value = $this->getDetailViewDisplayValue($value, $record, $uiType);
		}
		return $value;
	}

	/** {@inheritdoc} */
	public function getEditViewDisplayValue($value, $recordModel = false)
	{
		if (!empty($value)) {
			$this->edit = true;
			return $this->getDisplayValue($value);
		}
		return $value;
	}

	/**
	 * Function that converts the Number into Users Currency along with currency symbol.
	 *
	 * @param int|string $value
	 * @param int        $recordId
	 * @param int        $uiType
	 *
	 * @return Formatted Currency
	 */
	public function getDetailViewDisplayValue($value, $recordId, $uiType)
	{
		$moduleName = $this->getFieldModel()->getModuleName();
		if (!$moduleName) {
			$moduleName = \App\Record::getType($recordId);
		}
		if (72 === $uiType && $recordId && $currencyId = Currency::getCurrencyByModule($recordId, $moduleName)) {
			$currencySymbol = Currency::getById($currencyId)['currency_symbol'];
		} else {
			$userModel = User::getCurrentUserModel();
			$currencySymbol = $userModel->getDetail('currency_symbol');
		}
		return CurrencyField::appendCurrencySymbol($value, $currencySymbol);
	}

	/**
	 * Get default currency symbol.
	 *
	 * @param Vtiger_Record_Model|null $recordModel
	 *
	 * @return string
	 */
	public function getCurrencySymbol(?Vtiger_Record_Model $recordModel = null): string
	{
		$currencySymbol = User::getCurrentUserModel()->getDetail('currency_symbol');
		if ($recordModel && 72 === $this->getFieldModel()->getUIType()) {
			$currencyId = $recordModel->get('currency_id');
			if (!$currencyId && ($fieldModel = $recordModel->getField('currency_id')) && $fieldModel->isActiveField()) {
				$currencyId = $fieldModel->getDefaultFieldValue();
			}
			if ($currencyId && ($currencyData = Currency::getById($currencyId))) {
				$currencySymbol = $currencyData['currency_symbol'];
			}
		}

		return $currencySymbol;
	}

	/** {@inheritdoc} */
	public function getTemplateName()
	{
		return 'Edit/Field/Currency.tpl';
	}

	/** {@inheritdoc} */
	public function getAllowedColumnTypes()
	{
		return ['decimal'];
	}

	/** {@inheritdoc} */
	public function getQueryOperators()
	{
		return array_merge(['e', 'n', 'l', 'g', 'm', 'h', 'y', 'ny'], \App\Condition::FIELD_COMPARISON_OPERATORS);
	}

	/**
	 * Returns template for operator.
	 *
	 * @param string $operator
	 *
	 * @return string
	 */
	public function getOperatorTemplateName(string $operator = '')
	{
		return 'ConditionBuilder/Currency.tpl';
	}

	/** {@inheritdoc} */
	public function getDefaultOperator()
	{
		return 'a';
	}

	/** {@inheritdoc} */
	public function getDbConditionBuilderValue($value, string $operator)
	{
		$this->validate($value, true);
		preg_match_all('/[^\-\d]+/', $value, $matches);
		$matches[0] = array_map('trim', $matches[0]);
		if ($matches && $operators = array_intersect(array_map('App\\Purifier::decodeHtml', $matches[0]), IntegerField::$extendedOperators)) {
			$value = Purifier::decodeHtml($value);
			$valueConvert = [];
			$operators = array_values($operators);
			$explodeBySpace = explode(' ', $value);
			foreach ($explodeBySpace as $key => $valueToCondition) {
				$ev = explode($operators[$key], $valueToCondition);
				$valueConvert[] = $operators[$key] . $this->getDBValue($ev[1]) . '';
			}
			return implode(' ', $valueConvert);
		}
		return $this->getDBValue($value);
	}
}
