<?php
/**
 * Vat conversion data file.
 *
 * @package \App\Inventory
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mateusz Slominski <m.slominski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Inventory\Currency;

use App\Fields\Double;

/**
 * Vat conversion data class.
 */
class VatConversion implements \JsonSerializable, Conversion
{
	/**
	 * Construct.
	 *
	 * @param string $date
	 * @param string $value
	 * @param string $conversion
	 */
	public function __construct(
		public readonly string $date,
		public readonly string $value,
		public readonly string $conversion,
	) {}

	/** {@inheritDoc} */
	public static function fromArray(array $array): static
	{
		return new static($array['date'], (string) $array['value'], (string) $array['conversion']);
	}

	/** {@inheritDoc} */
	public static function getType(): string
	{
		return 'vat';
	}

	/** {@inheritDoc} */
	public function getDateObject(): \DateTimeInterface
	{
		return \DateTimeImmutable::createFromFormat('Y-m-d', $this->date);
	}

	/** {@inheritDoc} */
	public function jsonSerialize(): array
	{
		return $this->toArray();
	}

	/** {@inheritDoc} */
	public function toArray(): array
	{
		return [
			'date' => $this->date,
			'value' => $this->value,
			'conversion' => $this->conversion,
		];
	}

	/** {@inheritDoc} */
	public function getDateUserFormat(): string
	{
		return \DateTimeField::convertToUserFormat($this->date);
	}

	/** {@inheritDoc} */
	public function getValueUserFormat(): string
	{
		return Double::formatToDisplay($this->value, Double::FORMAT_USER_WITHOUT_ROUNDING);
	}
}
