<?php
/**
 * SMTP entity file.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Mail\Account\Entity;

use App\Encryption;
use App\Mail\Server;

/**
 * SMTP entity class.
 */
class SmtpEntity implements EntityInterface
{
	private readonly ?int $id;
	private readonly string $login;
	private ?string $refreshToken;
	private readonly string $authMethod;
	private readonly string $oauthProvider;
	private readonly string $clientId;
	private readonly string $clientSecret;
	private readonly int $redirectUriId;

	public function setData(
		#[\SensitiveParameter] array $data
	) {
		$this->id = $data['id'];
		$this->login = $data['username'];
		$this->authMethod = $data['auth_method'];
		$this->oauthProvider = $data['oauth_provider'];
		$this->clientId = $data['client_id'];
		$this->clientSecret = $data['client_secret'];
		$this->redirectUriId = $data['redirect_uri_id'];
		$this->refreshToken = Encryption::getInstance(Encryption::TARGET_SETTINGS)->decrypt($data['refresh_token'] ?? '');

		return $this;
	}

	public function set(string $key, mixed $value)
	{
		if (property_exists($this, $key)) {
			$this->{$key} = $value;
		}
	}

	public function get(string $key): mixed
	{
		return $this->{$key} ?? '';
	}

	public function getId(): ?int
	{
		return $this->id;
	}

	public function getLogin(): string
	{
		return $this->login;
	}

	public function setPassword(#[\SensitiveParameter] string $token)
	{
		return $this;
	}

	public function getPassword(): string
	{
		return '';
	}

	public function setExpireTime(string $expireTime): self
	{
		return $this;
	}

	public function getRefreshToken(): string
	{
		return $this->refreshToken;
	}

	public function setRefreshToken(#[\SensitiveParameter] string $token)
	{
		$this->refreshToken = $token;

		return $this;
	}

	public function getServer(): Server
	{
		return (new Server())
			->setAuthMethod($this->authMethod)
			->setClientId($this->clientId)
			->setOAuthProvider($this->oauthProvider)
			->setSecret($this->clientSecret)
			->setRedirectUriId($this->redirectUriId);
	}

	public function getValuesForSave(): array
	{
		$forSave = [];
		$forSave['refresh_token'] = Encryption::getInstance(Encryption::TARGET_SETTINGS)->encrypt($this->refreshToken);

		return $forSave;
	}

	public function setLogs(string $message)
	{
		return $this;
	}
}
