<?php
/**
 * Mail outlook message file.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

namespace App\Mail\Message;

use App\Mail\Account\Entity\AccountPrivateEntity;

/**
 * Mail outlook message class.
 */
class Outlook extends Imap
{
	/**
	 * Scanner engine name.
	 *
	 * @var string
	 */
	public $name = 'Outlook';

	private $mailAccount;

	/** {@inheritdoc} */
	public function getActions(): array
	{
		return array_filter(explode(',', \App\User::getCurrentUserModel()->getDetail('mail_scanner_actions')));
	}

	/** {@inheritdoc} */
	public function getMailCrmId(int $mailAccountId = 0)
	{
		if (!$this->mailCrmId) {
			$this->mailCrmId = \App\Mail\Message::findByCid($this->getUniqueId());
		}

		return $this->mailCrmId;
	}

	/** {@inheritdoc} */
	public function getUserId(): int
	{
		return \App\User::getCurrentUserRealId();
	}

	/**
	 * Initialize with request data.
	 *
	 * @param \App\Request $request
	 *
	 * @return void
	 */
	public function initFromRequest(\App\Request $request)
	{
		$this->set('message_id', $request->getByType('mailMessageId', 'MailId'))
			->set('date', $request->getByType('mailDateTimeCreated', 'DateTimeInIsoFormat'))
			->set('from', [$request->getByType('mailFrom', 'Email')])
			->set('to', $request->getArray('mailTo', 'Email'))
			->set('cc', $request->getArray('mailCc', 'Email'))
			->set('bcc', $request->getArray('mailBcc', 'Email'))
			->set('headers', $request->isEmpty('mailHeaders') ? '' : \App\TextUtils::textTruncate($request->getRaw('mailHeaders'), 16777215, false))
			->set('subject', $request->isEmpty('mailSubject') ? '-' : \App\TextUtils::textTruncate($request->getByType('mailSubject', 'Text'), 65535, false));
		$this->body = !$request->isEmpty('mailBody', true) ? $request->getForHtml('mailBody') : '';
	}

	/** {@inheritdoc} */
	public function getHeaderRaw(): string
	{
		return $this->get('headers');
	}

	/** {@inheritdoc} */
	public function getMsgUid(): int
	{
		return 0;
	}

	/** {@inheritdoc} */
	public function getFolderName(): string
	{
		return '';
	}

	/** {@inheritdoc} */
	public function hasAttachments(): bool
	{
		return false;
	}

	/** {@inheritdoc} */
	public function findRelatedRecords(bool $onlyId = false): array
	{
		$class = "App\\Mail\\ScannerAction\\CreatedMail";
		return (new $class())->setAccount($this->getAccount())->setMessage($this)->findRelatedRecords($onlyId);
	}

	/**
	 * Get related records.
	 *
	 * @return array
	 */
	public function getRelatedRecords(): array
	{
		$relations = [];
		$query = (new \App\Db\Query())->select(['vtiger_crmentity.crmid', 'vtiger_crmentity.setype'])
			->from('vtiger_ossmailview_relation')
			->innerJoin('vtiger_crmentity', 'vtiger_ossmailview_relation.crmid = vtiger_crmentity.crmid')
			->where(['vtiger_ossmailview_relation.ossmailviewid' => $this->getMailCrmId(0), 'vtiger_crmentity.deleted' => 0]);
		$dataReader = $query->createCommand()->query();
		while ($row = $dataReader->read()) {
			$relations[] = [
				'id' => $row['crmid'],
				'module' => $row['setype'],
				'label' => \App\Record::getLabel($row['crmid']),
			];
		}
		$dataReader->close();
		return $relations;
	}

	/** {@inheritdoc} */
	public function getExceptions(): array
	{
		return [];
	}

	public function getEmail(string $key): array
	{
		return $this->get($key) ?: [];
	}

	public function getAccount()
	{
		if (!$this->mailAccount) {
			$userModel = \App\User::getCurrentUserModel();
			$entity = new AccountPrivateEntity();
			$entity->set('owner_id', $userModel->getId())
				->setScannerActions($userModel->getDetail('mail_scanner_actions') ?? '')
				->setScannerFields($userModel->getDetail('mail_scanner_fields') ?? '')
				->setId(0);

			$mailAccount = new \App\Mail\Account();
			$mailAccount->setLogin('');
			$mailAccount->setSource($entity);

			$this->mailAccount = $mailAccount;
		}

		return $this->mailAccount;
	}

	/**
	 * Main function to execute scanner engine actions.
	 *
	 * @return void
	 */
	public function process(): void
	{
		$mailAccount = $this->getAccount();
		foreach ($mailAccount->getActions() as $action) {
			$class = "App\\Mail\\ScannerAction\\{$action}";
			(new $class())->setAccount($mailAccount)->setMessage($this)->process();
		}
	}
}
