<?php
/**
 * Base mail scanner action file.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

namespace App\Mail\ScannerAction;

/**
 * Base mail scanner action class.
 */
class CreatedMail extends Base
{
	/** {@inheritdoc} */
	public static $priority = 2;
	/** @var \App\Mail\Message\Imap */
	protected $message;

	/** {@inheritdoc} */
	public function process(): void
	{
		if ($this->checkExceptions()) {
			return;
		}

		$owner = $this->account->getSource()->getOwnerId();
		if ($mailCrmId = $this->message->getMailCrmId($this->account->getSource()->getId())) {
			$record = \OSSMailView_Record_Model::getInstanceById($mailCrmId, 'OSSMailView');
			$showners = $record->get('shownerid');
			if ($owner !== $record->get('assigned_user_id') && !\in_array($owner, explode(',', $showners))) {
				$fieldModel = $record->getField('shownerid');
				$sharedOwner = $showners ? explode(',', $showners) : [];
				$sharedOwner[] = $owner;
				$sharedOwner = implode(',', $sharedOwner);
				$record->set($fieldModel->getName(), $sharedOwner)->setDataForSave([$fieldModel->getTableName() => [$fieldModel->getColumnName() => $sharedOwner]])->save();
			}

			$this->message->setMailCrmId($record->getId());
			$this->message->setProcessData($this->getName(), ['mailViewId' => $record->getId()]);
			return;
		}

		$to = $this->message->getEmail('to');
		$from = $this->message->getEmail('from');
		$cc = $this->message->getEmail('cc');
		$bcc = $this->message->getEmail('bcc');
		$replyTo = $this->message->getEmail('reply_to');

		$record = \OSSMailView_Record_Model::getCleanInstance('OSSMailView');
		$record->set('assigned_user_id', $owner);
		$record->set('created_user_id', \App\User::getCurrentUserRealId());
		$record->setFromUserValue('subject', \App\TextUtils::textTruncate($this->message->getSubject(), $record->getField('subject')->getMaxValue(), false));
		$record->set('to_email', implode(',', $to));
		$record->set('from_email', implode(',', $from));
		$record->set('cc_email', implode(',', $cc));
		$record->set('bcc_email', implode(',', $bcc));
		$record->set('reply_to_email', implode(',', $replyTo));

		$record->set('date', $this->message->getDate());
		$record->set('createdtime', $this->message->getDate());
		$record->set('msgid', $this->message->getMsgId());
		$record->set('uid', $this->message->getMsgUid());
		$type = $this->message->getMailType($this->account);
		$record->set('type', $type);
		$record->set('rc_user', $this->account->getSource()->getId());
		$record->set('mbox', $this->message->getFolderName());
		$record->set('ossmailview_sendtype', \App\Mail\Message\Base::MAIL_TYPES[$type]);
		$record->set('orginal_mail', \App\TextUtils::htmlTruncate($this->message->getHeaderRaw(), $record->getField('orginal_mail')->getMaxValue()));
		$record->set('attachments_exist', (int) $this->message->hasAttachments());
		$record->setDataForSave(['vtiger_ossmailview' => ['cid' => $this->message->getUniqueId(), 'uid' => $this->message->getMsgUid()]]);

		if ($this->message->hasAttachments()) {
			$this->message->saveAttachments([
				'assigned_user_id' => $owner,
				'modifiedby' => $owner,
			]);
		}

		$record->set('content', \App\TextUtils::htmlTruncate($this->message->getBody(true), $record->getField('content')->getMaxValue()));
		$record->set('from_id', implode(',', array_unique(\App\Utils::flatten(\App\Mail\RecordFinder::getInstance()->setFields($this->getEmailsFields())->findByEmail($from)))));
		$toEmails = array_merge($to, $cc, $bcc);
		$record->set('to_id', implode(',', array_unique(\App\Utils::flatten(\App\Mail\RecordFinder::getInstance()->setFields($this->getEmailsFields())->findByEmail($toEmails)))));
		$record->save();

		$db = \App\Db::getInstance();
		foreach ($this->message->getDocuments() as $file) {
			$db->createCommand()->insert('vtiger_ossmailview_files', [
				'ossmailviewid' => $record->getId(),
				'documentsid' => $file['crmid'],
				'attachmentsid' => $file['attachmentsId'],
			])->execute();
		}

		$this->message->setMailCrmId($record->getId());
		$this->message->setProcessData($this->getName(), ['mailViewId' => $record->getId(), 'attachments' => $this->message->getDocuments()]);
	}
}
