<?php
/**
 * Service provides Query Generator ready to return data.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Łukasz Krawczyk <l.krawczyk@yetiforce.com>
 * @author Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Report\Provider;

use App\Config;
use App\Report\Builder\QueryGeneratorBuilder;
use App\Report\Decorator\ReportDataDecorator;
use App\Report\Enums\FunctionType;
use App\Report\Model\Expression;
use App\Report\Model\Query;
use App\Report\Repository\TemplateRepository;
use App\Report\Sanitizer\ReportSanitizer;
use App\User;

/** QueryGeneratorReportProvider class */
final class QueryGeneratorReportProvider implements ReportProvider
{
	public function __construct(
		private readonly TemplateRepository $templateRepository,
		private readonly QueryGeneratorBuilder $queryGeneratorBuilder,
		private readonly ReportDataDecorator $decorator,
		private readonly ReportSanitizer $sanitizer,
	) {}

	/**
	 * Provide data from report.
	 *
	 * @param int  $id
	 * @param int  $userId
	 * @param bool $isPreviewMode
	 *
	 * @throws \App\Exceptions\AppException
	 */
	public function provide(int $id, int $userId, bool $isPreviewMode = false): array
	{
		User::setCurrentUserId($userId);

		$template = $this->templateRepository->find($id);
		$query = $template->getQuery();

		$queryGenerator = $this->queryGeneratorBuilder->build($query, $userId);

		if (true === $isPreviewMode) {
			$queryGenerator->setLimit(Config::module('ReportTemplate', 'PREVIEW_ROW_LIMIT', 10));
		}

		$result = array_map(
			fn (array $item): array => $this->decorator->decorate($item, $query),
			$queryGenerator->createQuery()->all(),
		);

		if (true === $isPreviewMode) {
			$result = $this->roundForPreview($query, $result);
		}

		return $this->sanitizer->sanitize($query, $result);
	}

	private function roundForPreview(Query $query, array $data): array
	{
		$formulaAliases = array_map(
			fn (Expression $expression): string => $expression->getAlias(),
			array_filter(
				$query->getExpressions(),
				static fn(Expression $expression): bool => $expression->isFormula() === true
					&& in_array($expression->getFunction(), [FunctionType::PROD, FunctionType::QUO]),
			),
		);

		$result = [];
		foreach ($data as $key => $rowValue) {
			foreach ($rowValue as $alias => $value) {
				$result[$key][$alias] = (true === in_array($alias, $formulaAliases))
					? round($value, 2)
					: $value;
			}
		}

		return $result;
	}
}
