<?php
/**
 * Includes RelatedMembers relation.
 *
 * @package   Relation
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

use App\Db;
use App\Db\Query;
use App\User;

/**
 * Class GetRelatedMembers.
 */
class Occurrences_GetRelatedMembers_Relation extends Vtiger_GetRelatedList_Relation
{
	/** {@inheritdoc} */
	public const TABLE_NAME = 'u_#__relations_members_entity';

	/**
	 * Field custom list.
	 *
	 * @var array
	 */
	public array $customFields = [
		'status_rel' => [
			'label' => 'LBL_STATUS_REL',
			'uitype' => 16,
			'maximumlength' => 225
		],
		'comment_rel' => [
			'label' => 'LBL_COMMENT_REL',
			'uitype' => 21,
			'maximumlength' => 65535
		],
		'rel_created_user' => [
			'label' => 'LBL_RELATION_CREATED_USER',
			'uitype' => 52,
			'displaytype' => 10
		],
	];

	/**
	 * Field list.
	 *
	 * @param bool $editable
	 *
	 * @return array
	 */
	public function getFields(bool $editable = false): array
	{
		$fields = [];
		$sourceModule = $this->relationModel->getParentModuleModel();

		if ('Occurrences' !== $sourceModule->getName()) {
			$sourceModule = $this->relationModel->getRelationModuleModel();
		}

		foreach ($this->customFields as $fieldName => $data) {
			$field = new \Vtiger_Field_Model();

			$field->set('name', $fieldName)->set('column', $fieldName)
				->set('table', static::TABLE_NAME)
				->set('fromOutsideList', true)
				->setModule($sourceModule);

			foreach ($data as $key => $value) {
				$field->set($key, $value);
			}

			if (!$editable || !$field->isEditableReadOnly()) {
				$fields[$fieldName] = $field;
			}
		}

		return $fields;
	}

	/** {@inheritdoc} */
	public function getQuery(): void
	{
		parent::getQuery();

		$tableName = static::TABLE_NAME;
		$queryGenerator = $this->relationModel->getQueryGenerator();

		foreach (array_keys($this->customFields) as $fieldName) {
			$queryGenerator->setCustomColumn(["{$tableName}.{$fieldName}"]);
		}
	}

	/**
	 * {@inheritdoc}
	 *
	 * @throws \yii\db\Exception
	 */
	public function create(int $sourceRecordId, int $destinationRecordId): bool
	{
		if (!$this->hasRelationData($sourceRecordId, $destinationRecordId)) {
			return (bool)Db::getInstance()->createCommand()->insert(static::TABLE_NAME, [
				'crmid' => $sourceRecordId,
				'relcrmid' => $destinationRecordId,
				'rel_created_user' => User::getCurrentUserRealId(),
			])->execute();
		}

		return false;
	}

	/**
	 * UpdateRelationData function.
	 *
	 * @param int $sourceRecordId
	 * @param int $destinationRecordId
	 * @param array $updateData
	 *
	 * @return bool
	 * @throws \yii\db\Exception
	 */
	public function updateRelationData(int $sourceRecordId, int $destinationRecordId, array $updateData): bool
	{
		$conditions = [
			'or',
			['crmid' => $sourceRecordId, 'relcrmid' => $destinationRecordId],
			['crmid' => $destinationRecordId, 'relcrmid' => $sourceRecordId],
		];

		if ($this->hasRelationData($sourceRecordId, $destinationRecordId)) {
			return (bool) Db::getInstance()->createCommand()
				->update(static::TABLE_NAME, $updateData, $conditions)
				->execute();
		}

		return false;
	}

	/**
	 * Get relation data.
	 *
	 * @param int $sourceRecordId
	 * @param int $destinationRecordId
	 *
	 * @return array
	 */
	public function getRelationData(int $sourceRecordId, int $destinationRecordId): array
	{
		$data = (new Query())->from(static::TABLE_NAME)->where([
			'or',
			['crmid' => $sourceRecordId, 'relcrmid' => $destinationRecordId],
			['crmid' => $destinationRecordId, 'relcrmid' => $sourceRecordId],
		])->one();

		return $data ?: [];
	}

	private function hasRelationData(int $sourceRecordId, int $destinationRecordId): bool
	{
		$relationData = $this->getRelationData($sourceRecordId, $destinationRecordId);

		return !empty($relationData);
	}
}
