<?php

declare(strict_types=1);

use App\Db\Query;
use App\Exceptions\AppException;
use App\Report\Mapper\FilterMapper;
use App\Report\Model\Template;
use App\Report\Report;
use App\Report\Repository\TemplateRepository;
use App\Request;
use App\User;

/**
 * Record Class for ReportTemplate.
 *
 * @package   Model
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Klaudia Łozowska <k.Lozowska@yetiforce.com>
 */
final class ReportTemplate_Record_Model extends Vtiger_Record_Model
{
	/** @var array[] Fields restrictions for save action. */
	private const FIELDS = [
		self::QUERY_TABLE => [
			'input' => ['base_module_id', 'expressions', 'joins', 'filter_group'],
			'db' => ['base_module_id', 'parent_id', 'data_sources'],
		],
		self::QUERY_JOIN_TABLE => [
			'input' => ['relation_id', 'type'],
			'db' => ['query_id', 'relation_id', 'type'],
		],
		self::QUERY_EXPRESSION_TABLE => [
			'input' => ['field', 'function', 'group_by', 'order_by'],
			'db' => ['query_id', 'field', 'function', 'group_by', 'order_by', 'sort', 'formula_field'],
		],
		self::QUERY_FILTER_GROUP_TABLE => [
			'input' => ['condition', 'filters', 'filter_groups'],
			'db' => ['query_id', 'condition', 'parent_id'],
		],
		self::QUERY_FILTER_TABLE => [
			'input' => ['field', 'operator', 'value', 'subquery'],
			'db' => ['filter_group_id', 'field', 'operator', 'value', 'subquery_id'],
		],
	];

	/** @var string Report template query table name. */
	private const QUERY_TABLE = 'u_yf_report_template_query';

	/** @var string Report template query join table name. */
	private const QUERY_JOIN_TABLE = 'u_yf_report_template_query_join';

	/** @var string Report template query expression table name. */
	private const QUERY_EXPRESSION_TABLE = 'u_yf_report_template_query_expression';

	/** @var string Report template query filter group table name. */
	private const QUERY_FILTER_GROUP_TABLE = 'u_yf_report_template_query_filter_group';

	/** @var string Report template query filter table name. */
	private const QUERY_FILTER_TABLE = 'u_yf_report_template_query_filter';

	/**
	 * Array to store new report template query data.
	 *
	 * @var array
	 */
	private array $newQueryData = [];

	private static ?Template $domainTemplate = null;

	/**
	 * @inheritdoc
	 *
	 * @throws AppException
	 * @throws \yii\db\Exception
	 */
	public function saveToDb(): void
	{
		parent::saveToDb();
		if ($this->newQueryData) {
			$this->resetQuery();
		}
	}

	/** {@inheritdoc} */
	public function isEditable(): bool
	{
		return parent::isEditable() && $this->loggedUserIsReportOwner();
	}

	private function loggedUserIsReportOwner(): bool
	{
		return $this->get('assigned_user_id') === User::getCurrentUserId();
	}

	/**
	 * Store new query from request data.
	 *
	 * @param Request $request
	 *
	 * @return void
	 */
	public function setQueryDataFromRequest(Request $request): void
	{
		$this->newQueryData = $request->get('query');
	}

	/**
	 * Reset report template query.
	 *
	 * @return void
	 * @throws \yii\db\Exception|AppException
	 */
	private function resetQuery(): void
	{
		$db = \App\Db::getInstance();
		$createCommand = $db->createCommand();
		$createCommand->delete(
			self::QUERY_TABLE,
			['id' => $this->get('query') ? $this->get('query')['id'] : $this->get('temp_id')],
		)->execute();

		$queryId = $this->createQuery($this->newQueryData);

		$createCommand->update(
			$this->getEntity()->table_name,
			['query_id' => $queryId],
			['id' => $this->getId()],
		)->execute();
	}

	/**
	 * Create report template query.
	 *
	 * @param array $data
	 * @param int|null $parentId
	 *
	 * @return int
	 * @throws AppException
	 * @throws \yii\db\Exception
	 */
	private function createQuery(array $data, ?int $parentId = null): int
	{
		$queryId = $this->saveItem([...$data, 'parent_id' => $parentId], self::QUERY_TABLE);

		foreach ($data['expressions'] as $item) {
			$this->saveItem([...$item, 'query_id' => $queryId], self::QUERY_EXPRESSION_TABLE);
		}

		foreach ($data['joins'] as $item) {
			$this->saveItem([...$item, 'query_id' => $queryId], self::QUERY_JOIN_TABLE);
		}
		if(!empty($data['filter_group'])){
			$this->createFilterGroup($data['filter_group'], $queryId);
		}

		return $queryId;
	}

	/**
	 * Create report template query filter group.
	 *
	 * @param array $data
	 * @param int $queryId
	 *
	 * @return void
	 * @throws AppException
	 * @throws \yii\db\Exception
	 */
	private function createFilterGroup(array $data, int $queryId, int $parentId = null): void
	{
		$filterGroupId = $this->saveItem(
			[...$data, 'query_id' => $queryId, 'parent_id' => $parentId],
			self::QUERY_FILTER_GROUP_TABLE,
		);

		foreach ($data['filters'] as $filter) {
			if (true === array_key_exists('condition', $filter)) {
				$this->createFilterGroup($filter, $queryId, $filterGroupId);

				continue;
			}

			if (is_array($value = $filter['value'])) {
				$filter['value'] = json_encode($value);
			}

			$this->saveItem(
				[...$filter, 'filter_group_id' => $filterGroupId],
				self::QUERY_FILTER_TABLE,
			);
		}
	}

	/**
	 * Save item to database.
	 *
	 * @param array $data
	 * @param string $tableName
	 *
	 * @return int
	 * @throws AppException
	 * @throws \yii\db\Exception
	 */
	private function saveItem(array $data, string $tableName): int
	{
		$this->validateInputFields($data, $tableName);

		$db = \App\Db::getInstance();
		$createCommand = $db->createCommand();

		$data = array_intersect_key($data, array_flip(self::FIELDS[$tableName]['db']));

		$createCommand->insert($tableName, $data)->execute();

		return (int)$db->getLastInsertID($tableName);
	}

	/**
	 * Validate input fields.
	 *
	 * @param array $data
	 * @param string $tableName
	 *
	 * @return void
	 * @throws AppException
	 */
	private function validateInputFields(array $data, string $tableName): void
	{
		$context = match ($tableName) {
			self::QUERY_TABLE => 'query',
			self::QUERY_JOIN_TABLE => 'join',
			self::QUERY_EXPRESSION_TABLE => 'expression',
			self::QUERY_FILTER_GROUP_TABLE => 'filter_group',
			self::QUERY_FILTER_TABLE => 'filter',
			default => '',
		};

		foreach (self::FIELDS[$tableName]['input'] as $field) {
			if (!array_key_exists($field, $data)) {
			//	throw new AppException(sprintf('Missing value: %s:%s', $context, $field));
			}
		}
	}

	/**
	 * @inheritdoc
	 */
	protected function loadData($recordId): void
	{
		parent::loadData($recordId);

		if (!$this->get('query')) {
			self::$domainTemplate = Report::get(TemplateRepository::class)->find($recordId);

			if (null !== self::$domainTemplate) {
				$query = self::prepareQueryData(self::$domainTemplate->getQuery()->getId());
				$this->set('query', $query);
				$this->set('conditions', $query['filters']);
				$cacheName = "$recordId:{$this->getModule()->getName()}";
				\App\Cache::staticSave('RecordModel', $cacheName, $this);
			}
		}
	}

	/**
	 * Get report template query data.
	 *
	 * @param int $queryId
	 *
	 * @return array
	 */
	private static function prepareQueryData(int $queryId): array
	{
		$data = (new Query())
			->from('u_yf_report_template_query')
			->where(['id' => $queryId, 'parent_id' => null])
			->one();

		return [
			'id' => $data['id'],
			'base_module_id' => $data['base_module_id'],
			'data_sources' => $data['data_sources'],
			'base_module_name' => \App\Module::getModuleName($data['base_module_id']),
			'expressions' => self::prepareQueryExpressionsData($queryId),
			'joins' => self::prepareQueryJoinsData($queryId),
			'filters' => self::prepareQueryFiltersData(),
		];
	}

	/**
	 * Get report template query `join` data.
	 *
	 * @param int $queryId
	 *
	 * @return array
	 */
	private static function prepareQueryJoinsData(int $queryId): array
	{
		$dataReader = (new Query())
			->from('u_yf_report_template_query_join')
			->where(['query_id' => $queryId])
			->createCommand()
			->query();

		$data = [];
		while ($row = $dataReader->read()) {
			$data[$row['id']]['relation_id'] = $row['relation_id'];
			$data[$row['id']]['type'] = $row['type'];
		}

		return $data;
	}

	/**
	 * Get report template query `expression` data.
	 *
	 * @param int $queryId
	 *
	 * @return array
	 */
	private static function prepareQueryExpressionsData(int $queryId): array
	{
		$dataReader = (new Query())
			->from('u_yf_report_template_query_expression')
			->where(['query_id' => $queryId])
			->createCommand()
			->query();

		$data = [];
		while ($row = $dataReader->read()) {
			$data[$row['id']]['field'] = $row['field'];
			$data[$row['id']]['function'] = $row['function'];
			$data[$row['id']]['group_by'] = $row['group_by'];
			$data[$row['id']]['order_by'] = $row['order_by'];
			$data[$row['id']]['sort'] = $row['sort'];
			$data[$row['id']]['formula_field'] = $row['formula_field'];
		}

		return $data;
	}

	/**
	 * Get report template query `filter` data.
	 *
	 * @return array
	 */
	private static function prepareQueryFiltersData(): array
	{
		$filterGroup = self::$domainTemplate->getQuery()->getFilterGroup();

		return null !== $filterGroup ? Report::get(FilterMapper::class)->map($filterGroup) : [];
	}

	/**
	 * @inheritdoc
	 */
	public function isMandatorySave(): bool
	{
		return true;
	}

	public function getRecordListViewLinksRightSide(): array
	{
		$links = $recordLinks = [];

		$recordLinks[] = [
			'linktype' => 'LIST_VIEW_ACTIONS_RECORD_RIGHT_SIDE',
			'linklabel' => 'BTN_GENERATE_REPORT',
			'linkurl' => sprintf('javascript:ReportTemplate_List_Js.generateReport(%d)', $this->getId()),
			'linkicon' => 'fas fa-download',
			'linkclass' => 'btn-sm btn-primary',
		];

		foreach ($recordLinks as $recordLink) {
			$links[] = Vtiger_Link_Model::getInstanceFromValues($recordLink);
		}

		return $links;
	}

	public function getReportFieldModel(string $expression)
	{
		[$fieldName, $fieldModuleName, $sourceFieldName] = array_pad(explode(':', $expression), 3, false);
		if(false === $fieldModuleName){
			return Vtiger_Field_Model::getInstance($fieldName, Vtiger_Module_Model::getInstance($this->get('query')['base_module_name']));
		}
		if('INVENTORY'=== $sourceFieldName){
			return	Vtiger_Inventory_Model::getInstance($fieldModuleName)->getField($fieldName);
		}
		return Vtiger_Field_Model::getInstance($fieldName, Vtiger_Module_Model::getInstance($fieldModuleName));
	}
}
