<?php

/**
 * Report filter group model.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Leszek Koziatek <l.koziatek@yetiforce.com>
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Report\Model;

use App\Report\Enums\ConditionType;

final class FilterGroup
{
	/**
	 * @var array<Filter>
	 */
	private array $filters = [];

	/** @var array<FilterGroup> */
	private array $children = [];

	public function __construct(
		private int $id,
		private Query $query,
		private ConditionType $condition,
		private ?self $parent = null,
	) {
		$this->parent?->addChild($this);
	}

	/** Get module name */
	public function getModuleName(): string
	{
		return $this->query->getModuleName();
	}

	/** Get filter group id */
	public function getId(): int
	{
		return $this->id;
	}

	/** Get related query */
	public function getQuery(): Query
	{
		return $this->query;
	}

	/** GEt related condition */
	public function getCondition(): ConditionType
	{
		return $this->condition;
	}

	/** Get parent filter group */
	public function getParent(): ?self
	{
		return $this->parent;
	}

	/** Get filters in group */
	public function getFilters(): array
	{
		return $this->filters;
	}

	/** Add filter to group */
	public function addFilter(Filter $filter): self
	{
		if (false === \array_key_exists($id = $filter->getId(), $this->filters)) {
			$this->filters[$id] = $filter;
		}

		return $this;
	}

	/** Get children for Filter Group */
	public function getChildren(): array
	{
		return $this->children;
	}

	/** Get children Filter Group IDs */
	public function getChildrenIds(bool $recursive = false): array
	{
		$ids = [];
		foreach ($this->children as $childFilterGroup) {
			$ids[] = $childFilterGroup->getId();
			if ($recursive) {
				$ids += $childFilterGroup->getChildrenIds($recursive);
			}
		}

		return $ids;
	}

	/** Add child Filter Group */
	public function addChild(self $filterGroup): self
	{
		if (false === \array_key_exists($id = $filterGroup->getId(), $this->children)) {
			$this->children[$id] = $filterGroup;
		}

		return $this;
	}

	/** Seek for Filter Group id in current Filter Group */
	public function getChild(int $id): ?self
	{
		$foundFilterGroup = null;
		foreach ($this->children as $child) {
			$foundFilterGroup = $child->getId() === $id ? $child : $child->getChild($id);
			if ($foundFilterGroup) {
				break;
			}
		}

		return $foundFilterGroup;
	}
}
