<?php

namespace App\Conditions\QueryFields;

/**
 * Multipicklist Query Field Class.
 *
 * @package UIType
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Tomasz Kur <t.kur@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */
class MultipicklistField extends BaseField
{
	/** @var string Separator. */
	protected $separator = ' |##| ';

	/** @var string Condition separator. */
	protected $conditionSeparator = '##';

	/** {@inheritdoc} */
	public function getValue()
	{
		return explode($this->conditionSeparator, $this->value);
	}

	/**
	 * Function to get combinations of string from Array.
	 * To be used for small sets, as the result depends on n! and may lead to memory exhaustion.
	 *
	 * @param array  $array
	 * @param string $tempString
	 *
	 * @return array
	 */
	public function getCombinations($array, $tempString = '')
	{
		$countArray = \count($array);
		$result = '';
		for ($i = 0; $i < $countArray; ++$i) {
			$splicedArray = $array;
			$element = array_splice($splicedArray, $i, 1);
			if (\count($splicedArray) > 0) {
				if (!\is_array($result)) {
					$result = [];
				}
				$result = array_merge($result, $this->getCombinations($splicedArray, $tempString . $this->separator . $element[0]));
			} else {
				return [$tempString . $this->separator . $element[0]];
			}
		}
		return $result;
	}

	/** {@inheritdoc} */
	public function operatorE(): array
	{
		$values = $this->getValue();
		// If there are less than 5 values, we can use combinations to optimize the query.
		if (\count($values) < 5) {
			$values = array_map(fn ($value) => ltrim($value, $this->separator), $this->getCombinations($values));
			$condition = [$this->getColumnName() => $values];
		} else {
			$condition = $this->getConditionsForMultiValues($values);
		}

		return $condition;
	}

	/** {@inheritdoc} */
	public function operatorN(): array
	{
		$values = $this->getValue();
		// If there are less than 5 values, we can use combinations to optimize the query.
		if (\count($values) < 5) {
			$values = array_map(fn ($value) => ltrim($value, $this->separator), $this->getCombinations($values));
			$condition = ['not', [$this->getColumnName() => $values]];
		} else {
			$condition = ['not', $this->getConditionsForMultiValues($values)];
		}

		return $condition;
	}

	/** {@inheritdoc} */
	public function operatorC(): array
	{
		$condition = ['or'];
		foreach ($this->getValue() as $value) {
			array_push($condition, [$this->getColumnName() => $value], ['or like', $this->getColumnName(),
				[
					"%{$this->separator}{$value}{$this->separator}%",
					"{$value}{$this->separator}%",
					"%{$this->separator}{$value}",
				], false,
			]);
		}
		return $condition;
	}

	/** {@inheritdoc} */
	public function operatorK(): array
	{
		$condition = ['and'];
		foreach ($this->getValue() as $value) {
			array_push($condition, ['<>', $this->getColumnName(), $value], ['not', ['or like', $this->getColumnName(),
				[
					"%{$this->separator}{$value}{$this->separator}%",
					"{$value}{$this->separator}%",
					"%{$this->separator}{$value}",
				], false,
			]]);
		}
		return $condition;
	}

	/**
	 * Get conditions for multiple values.
	 * This method constructs a condition that checks if the column contains all specified values,
	 * and is used for operators 'e' (equals) and 'n' (not equals).
	 *
	 * @param array $values
	 *
	 * @return string[]
	 */
	protected function getConditionsForMultiValues(array $values): array
	{
		$column = $this->getColumnName();
		$condition = ['and'];
		foreach ($values as $value) {
			$condition[] = ['or like', $column,
				[
					"%{$this->separator}{$value}{$this->separator}%",
					"{$value}{$this->separator}%",
					"%{$this->separator}{$value}",
				], false,
			];
		}
		$countExpr = "(LENGTH($column) - LENGTH(REPLACE($column, '{$this->separator}', ''))) / " . \strlen($this->separator);
		$condition[] = ['=', $countExpr, \count($values) - 1];

		return $condition;
	}
}
