<?php
/**
 * Doctrine migrator file.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Klaudia Łozowska <k.lozowska@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

declare(strict_types=1);

namespace App\Db\Migrations\Doctrine;

use App\Db\Migrations\MigratorInterface;
use Config\Db;
use Doctrine\DBAL\DriverManager;
use Doctrine\Migrations\Configuration\Connection\ExistingConnection;
use Doctrine\Migrations\Configuration\Migration\PhpFile;
use Doctrine\Migrations\DependencyFactory;
use Doctrine\Migrations\Exception\NoMigrationsFoundWithCriteria;
use Doctrine\Migrations\Exception\NoMigrationsToExecute;
use Doctrine\Migrations\Finder\Exception\InvalidDirectory;
use Doctrine\Migrations\Metadata\MigrationPlanList;
use Doctrine\Migrations\MigratorConfiguration;

/**
 * Doctrine migrator class.
 */
class Migrator implements MigratorInterface
{
	private const MIGRATIONS_CONFIG_FILENAME = 'db-migrations-config.php';

	private DependencyFactory $dependencyFactory;

	public function __construct()
	{
		$this->dependencyFactory = DependencyFactory::fromConnection(
			new PhpFile(self::MIGRATIONS_CONFIG_FILENAME),
			new ExistingConnection(DriverManager::getConnection([
				'dbname' => Db::$db_name,
				'user' => Db::$db_username,
				'password' => Db::$db_password,
				'host' => Db::$db_server,
				'port' => Db::$db_port,
				'driver' => 'pdo_mysql',
			])),
		);
		$this->dependencyFactory->getMetadataStorage()->ensureInitialized();
	}

	/**
	 * {@inheritDoc}
	 */
	public function migrate(): void
	{
		$migratorConfiguration = (new MigratorConfiguration())
			->setDryRun(false)
			->setTimeAllQueries(false)
			->setAllOrNothing($this->dependencyFactory->getConfiguration()->isAllOrNothing());

		$this->dependencyFactory->getMigrator()->migrate($this->getMigrationPlan(), $migratorConfiguration);
	}

	/** {@inheritdoc} */
	public function pendingMigrationsCount(): int
	{
		try {
			$count = $this->getMigrationPlan()->count();
		} catch (NoMigrationsToExecute|NoMigrationsFoundWithCriteria|InvalidDirectory) {
			$count = 0;
		}

		return $count;
	}

	/**
	 * Get migration plan.
	 *
	 * @return MigrationPlanList
	 */
	private function getMigrationPlan(): MigrationPlanList
	{
		$planCalculator = $this->dependencyFactory->getMigrationPlanCalculator();

		return $planCalculator->getPlanUntilVersion($planCalculator->getMigrations()->getLast()->getVersion());
	}
}
