<?php
/**
 * Autologin repository file.
 *
 * @package \App\Mail
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Mail\Account\Repository;

use App\Db;
use App\Mail\Account\Entity\AccountPrivateEntity;
use App\Mail\Account\Entity\Enum\MailBoxType;
use App\Mail\Account\Entity\Enum\Status;

/**
 * Autologin repository class.
 */
class AutologinRepository implements AccountRepositoryInterface
{
	protected const TABLE_NAME = 'roundcube_users_autologin';
	protected const ACCOUNT_TABLE_NAME = 's_#__mail_account';

	private readonly Db $db;

	public function __construct()
	{
		$this->db = Db::getInstance('admin');
	}

	/**
	 * Return the ID of all users who can log in to this account, no private accounts.
	 *
	 * @param int $id
	 *
	 * @return array
	 */
	public function findById(int $id): array
	{
		return (new \App\Db\Query())->select(['crmuser_id'])->from(self::TABLE_NAME)->where(['rcuser_id' => $id])->column() ?: [];
	}

	/**
	 * Return the ID of all users who can log in to this account, including private accounts or not.
	 *
	 * @param int  $userId
	 * @param bool $includePrivete
	 *
	 * @return array
	 */
	public function getAllByUserId(int $userId, bool $includePrivete): array
	{
		$response = [];
		$autologinTable = self::TABLE_NAME;

		$conditions = ['or',
			['private' => MailBoxType::PUBLIC->value, 'crmuser_id' => $userId]
		];
		if ($includePrivete) {
			$conditions[] = ['private' => MailBoxType::PRIVATE->value, 'owner_id' => $userId];
		}

		$dataReader = (new \App\Db\Query())->select(self::ACCOUNT_TABLE_NAME . '.*')->from(self::ACCOUNT_TABLE_NAME)
			->leftJoin($autologinTable, self::ACCOUNT_TABLE_NAME . ".id = {$autologinTable}.rcuser_id")
			->where(['status' => Status::STATUS_ACTIVE->value])
			->andWhere($conditions)
			->orderBy(['crmuser_id' => SORT_ASC, "{$autologinTable}.active" => SORT_DESC])
			->createCommand()->query();
		while ($row = $dataReader->read()) {
			$response[$row['id']] = (new AccountPrivateEntity())->setData(...$row);
		}

		return $response;
	}

	/**
	 * Seve autologin to account.
	 *
	 * @param int   $accountId
	 * @param array $users
	 *
	 * @return void
	 */
	public function save(int $accountId, array $users)
	{
		$insert = [];
		foreach ($users as $user) {
			$insert[] = [$accountId, $user];
		}

		$transaction = $this->db->beginTransaction();
		try {
			$this->db->createCommand()->delete(self::TABLE_NAME, ['rcuser_id' => $accountId])->execute();
			if ($insert) {
				$this->db->createCommand()->batchInsert(self::TABLE_NAME, ['rcuser_id', 'crmuser_id'], $insert)->execute();
			}
			$transaction->commit();
		} catch (\Throwable $ex) {
			$transaction->rollBack();
			throw $ex;
		}
	}
}
