<?php

/**
 * Mail imap file.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

namespace App\Mail\Connections;

use App\Exceptions\AppException;
use Webklex\PHPIMAP\Client;
use Webklex\PHPIMAP\ClientManager;
use Webklex\PHPIMAP\Folder;
use Webklex\PHPIMAP\Support\MessageCollection;

/**
 * Mail imap class.
 */
class Imap
{
	/** @var bool Debug */
	public bool $debug = false;
	/** @var string Server hostname. */
	protected $host;
	/** @var int Server port. */
	protected $port;
	/** @var string Server encryption. Supported: none, ssl, tls, starttls or notls. */
	protected $encryption;
	/** @var string Server protocol. */
	protected $protocol = 'imap';
	/** @var bool Validate cert. */
	protected $validateCert = true;
	/** @var int Connection timeout. */
	protected $timeout = 15;
	/** @var string Account username. */
	protected $username;
	/** @var string Account password. */
	protected $password;
	/**
	 * Account authentication method.
	 *
	 * @var string|null
	 *
	 * @example oauth, null
	 */
	protected $authentication;

	/** @var Client */
	private $client;

	/** @var int */
	private $attempt = 0;

	/** @var string Stores error messages related to IMAP connections. */
	private string $error = '';

	/**
	 * Constructor.
	 *
	 * @param array $options
	 */
	public function __construct(array $options)
	{
		$reflect = new \ReflectionClass($this);
		foreach ($options as $name => $value) {
			if ($reflect->hasProperty($name) && !$reflect->getProperty($name)->isPrivate()) {
				$this->{$name} = $value;
			}
		}
	}

	/**
	 * Get the latest error.
	 *
	 * @return string
	 */
	public function getError(): string
	{
		return $this->error;
	}

	/**
	 * Reports an error number and string.
	 *
	 * @param int    $errno   The error number returned by PHP
	 * @param string $errmsg  The error message returned by PHP
	 * @param string $errfile The file the error occurred in
	 * @param int    $errline The line number the error occurred on
	 */
	public function errorHandler($errno, $errmsg, $errfile = '', $errline = 0)
	{
		$notice = 'IMAP Connection failed.';
		if ($this->debug) {
			$this->error = "{$notice} Error #{$errno}: {$errmsg} [{$errfile} line {$errline}]";
		} else {
			$this->error = "{$notice} Error #{$errno}: {$errmsg}";
		}
	}

	/**
	 * Connect to server.
	 *
	 * @return $this
	 */
	public function connect()
	{
		if (!$this->client || !$this->client->isConnected()) {
			$this->client = (new ClientManager(['options' => ['debug' => $this->debug]]))->make([
				'host' => $this->host,
				'port' => $this->port,
				'encryption' => $this->encryption, // 'ssl',
				'validate_cert' => $this->validateCert,
				'protocol' => $this->protocol,
				'authentication' => $this->authentication,
				'username' => $this->username,
				'password' => $this->password,
				'timeout' => $this->timeout
			]);
			++$this->attempt;

			try {
				set_error_handler([$this, 'errorHandler']);
				$this->client->connect();
				restore_error_handler();
			} catch (\Throwable $th) {
				restore_error_handler();
				throw $th;
			}
		}

		return $this;
	}

	public function isConnected()
	{
		return $this->client && $this->client->isConnected();
	}

	/**
	 * Disconnect from server.
	 *
	 * @return $this
	 */
	public function disconnect()
	{
		if ($this->client && $this->client->isConnected()) {
			$this->client->disconnect();
		}

		return $this;
	}

	/**
	 * Get folders list.
	 * If hierarchical order is set to true, it will make a tree of folders, otherwise it will return flat array.
	 *
	 * @param bool        $hierarchical
	 * @param string|null $parentFolder
	 */
	public function getFolders(bool $hierarchical = true, ?string $parentFolder = null)
	{
		$this->connect();
		$folders = [];

		foreach ($this->client->getFolders($hierarchical, $parentFolder) as $folder) {
			$folders = $this->getChildrenFolders($folder, $folders);
		}

		return $folders;
	}

	/**
	 * Get mailbox folder.
	 *
	 * @param string $name Folder name (utf8)
	 *
	 * @return Folder|null
	 */
	public function getFolderByName(string $name): ?Folder
	{
		$this->connect();

		$folder = $this->client->getFolderByPath($name);

		/* @see https://github.com/mjl-/mox/issues/80#issuecomment-1759489304 */
		if (!$folder && 'INBOX' === strtoupper($name)) {
			$name = ctype_upper($name) ? ucfirst(strtolower($name)) : strtoupper($name);
			$folder = $this->client->getFolderByPath($name);
		}

		return $folder;
	}

	/**
	 * Get all messages with an uid greater than a given UID.
	 *
	 * @param int    $uid
	 * @param string $folderName
	 * @param int    $limit
	 *
	 * @return MessageCollection
	 */
	public function getMessagesGreaterThanUid(string $folderName, int $uid, int $limit): MessageCollection
	{
		return $this->getFolderByName($folderName)->query()->limit($limit)->getByUidGreater($uid);
	}

	public function getMessageByUid(string $folderName, int $uid): ?\App\Mail\Message\Imap
	{
		$message = $this->getFolderByName($folderName)?->query()->whereUid($uid)->get()->first();

		return $message ? (new \App\Mail\Message\Imap())->setMessage($message) : null;
	}

	public function getLastMessages(int $limit = 5, string $folderName = 'INBOX')
	{
		$folder = $this->getFolderByName($folderName);
		$messages = [];
		if ($folder) {
			foreach ($folder->query()->setFetchOrder('desc')->limit($limit)->all()->get()->reverse() as $message) {
				$messages[$message->getUid()] = (new \App\Mail\Message\Imap())->setMessage($message);
			}
		}

		return $messages;
	}

	/**
	 * Append a text message to the mailbox of the specified folder.
	 *
	 * @param string $folderName
	 * @param string $message
	 * @param array  $options
	 *
	 * @return array
	 */
	public function appendMessage(string $folderName, string $message, array $options = []): array
	{
		$this->connect();
		$folder = $this->client->getFolder($folderName);
		if (!$folder) {
			throw new AppException('ERR_IMAP_FOLDER_NOT_EXISTS||' . $folderName);
		}

		return $folder->appendMessage($message, $options);
	}

	/**
	 * Get children folders.
	 *
	 * @param object $folder
	 * @param array  $folders
	 *
	 * @return array
	 */
	private function getChildrenFolders($folder, array &$folders): array
	{
		$folders[$folder->full_name] = [
			'name' => $folder->name,
			'fullName' => $folder->full_name,
			// Indicates if folder is only container, not a mailbox - you can't open it.
			'noSelect' => $folder->no_select ?? false,
		];
		if ($folder->hasChildren()) {
			$folders[$folder->full_name]['children'] = [];
			foreach ($folder->children as $subFolder) {
				$this->getChildrenFolders($subFolder, $folders[$folder->full_name]['children']);
			}
		}

		return $folders;
	}
}
