<?php
/**
 * Base class to handle map drivers.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Klaudia Łozowska <k.lozowska@yetiforce.com>
 */

declare(strict_types=1);

namespace App\Map;

use App\Db;
use App\Encryption;
use yii\db\Exception;

/**
 * Base class to handle map drivers.
 */
abstract class BaseConnector
{
	/**
	 * Map driver configuration table name.
	 *
	 * @var string
	 */
	protected static string $configTableName;

	/**
	 * Active driver instance.
	 *
	 * @var ?BaseDriver
	 */
	private static ?BaseDriver $activeInstance = null;

	/**
	 * Get driver class name.
	 *
	 * @param string $driver
	 *
	 * @return string
	 */
	abstract static function getDriverClass(string $driver): string;

	/**
	 * Function to get connector.
	 *
	 * @param string $driver
	 *
	 * @return BaseDriver
	 */
	public static function getInstance(string $driver): BaseDriver
	{
		$className = static::getDriverClass($driver);

		return new $className(static::getConfig($driver));
	}

	/**
	 * Function to get active connector.
	 *
	 * @return ?BaseDriver
	 */
	public static function getActiveInstance(): ?BaseDriver
	{
		if (static::$activeInstance) {
			return static::$activeInstance;
		}

		$config = static::getActiveConfig();
		if (!$config) {
			return null;
		}

		$className = static::getDriverClass($config['driver']);

		return static::$activeInstance = new $className($config);
	}

	/**
	 * Switch active driver.
	 *
	 * @param string $driver
	 *
	 * @throws Exception
	 */
	public static function switchActive(string $driver): void
	{
		Db::getInstance()
			->createCommand()
			->update(static::$configTableName, ['active' => 0])
			->execute();

		Db::getInstance()
			->createCommand()
			->update(static::$configTableName, ['active' => 1], ['driver' => $driver,])
			->execute();
	}

	/**
	 * Save configuration.
	 *
	 * @param string $driverName
	 * @param array $config
	 *
	 * @return void
	 * @throws Exception
	 */
	public function saveConfig(string $driverName, array $config): void
	{
		Db::getInstance()
			->createCommand()
			->update(static::$configTableName, $config, ['driver' => $driverName])
			->execute();
	}

	/**
	 * Get all map coordinates drivers configurations.
	 *
	 * @return array
	 */
	public static function getConfigs(): array
	{
		return (new Db\Query())
			->from(static::$configTableName)
			->all();
	}

	/**
	 * Get map coordinates driver configuration.
	 *
	 * @param string $driver
	 *
	 * @return array
	 */
	public static function getConfig(string $driver): array
	{
		return (new Db\Query())
			->from(static::$configTableName)
			->where(['driver' => $driver])
			->one();
	}

	/**
	 * Get map coordinates active driver configuration.
	 *
	 * @return ?array
	 */
	private static function getActiveConfig(): ?array
	{
		return (new Db\Query())
			->from(static::$configTableName)
			->where(['active' => 1])
			->one() ?: null;
	}
}
