<?php
/**
 * Google Maps driver file to get tile layer.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Klaudia Łozowska <k.lozowska@yetiforce.com>
 *
 * @see       https://developers.google.com/maps/documentation/javascript/overview
 */

declare(strict_types=1);

namespace App\Map\TileLayer;

/**
 * Connector for service Google Maps to get tile layer.
 */
class GoogleMaps extends Base
{
	/**
	 * @inheritdoc
	 */
	protected string $label = 'LBL_TILE_LAYER_GOOGLE_MAPS';

	/**
	 * @inheritdoc
	 */
	protected string $url = 'https://developers.google.com/maps/documentation';

	/**
	 * @inheritdoc
	 */
	protected string $urlTemplate = 'https://maps.googleapis.com/maps/api/js';

	/**
	 * @inheritdoc
	 */
	protected array $formFields = [
		'api_key' => [
			'validator' => [['name' => 'AlphaNumeric']],
			'uitype' => 99,
			'label' => 'LBL_API_KEY',
			'purifyType' => \App\Purifier::ALNUM_EXTENDED,
			'maximumlength' => '200',
			'typeofdata' => 'V~M',
		],
	];

	/**
	 * @inheritDoc
	 */
	public function getUrlTemplate(): string
	{
		$key = $this->getConfig()['api_key'] ?? null;

		return $this->urlTemplate . "?key=$key";
	}

	/**
	 * @inheritdoc
	 */
	public function getAssetsDomains(): array
	{
		return [
			'allowedImageDomains' => ['maps.googleapis.com', 'maps.gstatic.com'],
			'allowedScriptDomains' => ['maps.googleapis.com'],
			'allowedConnectDomains' => ['maps.googleapis.com'],
			'allowedStylesheetDomains' => ['fonts.googleapis.com'],
			'allowedFontDomains' => ['fonts.gstatic.com'],
		];
	}
}
