<?php
/**
 * ModTracker data access implementation for relational DB file.
 *
 * @package \App\ModTracker
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mateusz Slominski <m.slominski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\ModTracker\Repository\Relational;

use App\Db\Query;
use App\ModTracker\Repository\ModTrackerRepositoryInterface;

/**
 * ModTracker data access implementation for relational DB class.
 */
class ModTrackerRepository implements ModTrackerRepositoryInterface
{
	/**
	 * Constructor.
	 *
	 * @param \App\Db $db
	 */
	public function __construct(
		private \App\Db $db
	) {
	}

	/**
	 * {@inheritDoc}
	 */
	public function createEntry(array $entry, array $changeDetails = []): string
	{
		if (isset($entry['last_reviewed_users'])) {
			$entry['last_reviewed_users'] = '#' . (\is_array($entry['last_reviewed_users']) ? implode('#', $entry['last_reviewed_users']) : $entry['last_reviewed_users']) . '#';
		}

		$this->db->createCommand()->insert('vtiger_modtracker_basic', $entry)->execute();
		$id = $this->db->getLastInsertID('vtiger_modtracker_basic_id_seq');

		if ($changeDetails) {
			$batchInsert = [];
			foreach ($changeDetails as $fieldName => $detail) {
				$batchInsert[] = [$id, $fieldName, $detail['prevalue'], $detail['postvalue']];
			}

			$this->db->createCommand()
				->batchInsert('vtiger_modtracker_detail', ['id', 'fieldname', 'prevalue', 'postvalue'], $batchInsert)
				->execute();
		}

		return $id;
	}

	/**
	 * {@inheritDoc}
	 */
	public function createInventoryEntry(array $entry): string
	{
		$entry['changes'] = \App\Json::encode($entry['changes']);
		$this->db->createCommand()->insert('u_#__modtracker_inv', $entry)->execute();

		return $this->db->getLastInsertID('u_#__modtracker_inv_id_seq');
	}

	/**
	 * {@inheritDoc}
	 */
	public function createRelationEntry(array $entry): string
	{
		$this->db->createCommand()->insert('vtiger_modtracker_relations', $entry)->execute();

		return $this->db->getLastInsertID('vtiger_modtracker_relations_id_seq');
	}

	/**
	 * {@inheritDoc}
	 */
	public function getUpdates(int $parentRecordId, \Vtiger_Paging_Model $pagingModel, string $type, ?int $startWith = null)
	{
		$startIndex = $pagingModel->getStartIndex();
		$pageLimit = $pagingModel->getPageLimit();
		$where = self::getConditionByType($type);
		$query = (new Query())
			->from('vtiger_modtracker_basic')
			->where(['crmid' => $parentRecordId])
			->andWhere($where)
			->limit($pageLimit)
			->offset($startIndex)
			->orderBy(['changedon' => SORT_DESC]);
		if (!empty($startWith)) {
			$query->andWhere(['>=', 'id', $startWith]);
		}
		$dataReader = $query->createCommand()->query();
		while ($row = $dataReader->read()) {
			$row['last_reviewed_users'] = array_map('intval', array_filter(explode('#', $row['last_reviewed_users'])));
			yield $row;
		}
		$dataReader->close();
	}

	/**
	 * {@inheritDoc}
	 */
	public function setLastReviewed(int $userId, int $recordId, int $status)
	{
		$row = (new Query())->select(['last_reviewed_users', 'id'])
			->from('vtiger_modtracker_basic')
			->where(['crmid' => $recordId])
			->andWhere(['<>', 'status', $status])
			->orderBy(['changedon' => SORT_DESC, 'id' => SORT_DESC])
			->limit(1)
			->one();

		if ($row) {
			$lastReviewedUsers = explode('#', $row['last_reviewed_users']);
			$lastReviewedUsers[] = $userId;
			\App\Db::getInstance()->createCommand()
				->update('vtiger_modtracker_basic', ['last_reviewed_users' => '#' . implode('#', array_filter($lastReviewedUsers)) . '#'], ['id' => $row['id']])
				->execute();

			return $row['id'];
		}
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	public function findRelationById(string|int $id): array
	{
		return (new Query())->from('vtiger_modtracker_relations')->where(['id' => (int) $id])->one();
	}

	/**
	 * {@inheritDoc}
	 */
	public function getRelationsByIds(array $ids): array
	{
		return (new Query())->from('vtiger_modtracker_relations')->where(['in', 'id', $ids])->all();
	}

	/**
	 * {@inheritDoc}
	 */
	public function unsetReviewed(int $userId, int $recordId, int $statusException, int|string $entryException)
	{
		$query = new Query();
		$query->select(['last_reviewed_users', 'id'])
			->from('vtiger_modtracker_basic')
			->where(['crmid' => $recordId])
			->andWhere(['<>', 'status', $statusException])
			->andWhere(['like', 'last_reviewed_users', "#$userId#"])
			->orderBy(['changedon' => SORT_DESC, 'id' => SORT_DESC])
			->limit(1);

		if ($entryException) {
			$query->andWhere(['<>', 'id', (int) $entryException]);
		}
		$row = $query->one();
		if ($row) {
			$lastReviewedUsers = array_filter(explode('#', $row['last_reviewed_users']));
			$key = array_search($userId, $lastReviewedUsers);
			unset($lastReviewedUsers[$key]);
			$value = empty($lastReviewedUsers) ? '' : '#' . implode('#', array_filter($lastReviewedUsers)) . '#';

			return $this->db->createCommand()->update('vtiger_modtracker_basic', ['last_reviewed_users' => $value], ['id' => $row['id']])->execute();
		}
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	public function getLastReviewedUsers(int $recordId, int $status): array
	{
		$lastReviewedUsers = (new Query())->select(['last_reviewed_users'])->from('vtiger_modtracker_basic')
			->where(['crmid' => $recordId])
			->andWhere(['<>', 'status', $status])
			->orderBy(['changedon' => SORT_DESC, 'id' => SORT_DESC])
			->scalar();

		return $lastReviewedUsers ? array_filter(explode('#', $lastReviewedUsers)) : [];
	}

	/**
	 * {@inheritDoc}
	 */
	public function getTotalRecordCount(int $recordId, ?string $type = null): int
	{
		return (new Query())
			->from('vtiger_modtracker_basic')
			->where(['crmid' => $recordId])
			->andWhere($this->getConditionByType($type))
			->count();
	}

	/**
	 * {@inheritDoc}
	 */
	public function getFieldHistory(int $record, string $fieldName): array
	{
		$rows = [];
		$query = (new Query())
			->select(['vtiger_modtracker_basic.changedon', 'vtiger_modtracker_detail.prevalue', 'vtiger_modtracker_detail.postvalue'])
			->from('vtiger_modtracker_detail')
			->leftJoin('vtiger_modtracker_basic', 'vtiger_modtracker_detail.id = vtiger_modtracker_basic.id')
			->where(['vtiger_modtracker_basic.crmid' => $record, 'vtiger_modtracker_detail.fieldname' => $fieldName])->orderBy(['vtiger_modtracker_basic.id' => SORT_ASC]);
		$dataReader = $query->createCommand()->query();
		while ($row = $dataReader->read()) {
			$row['detail'][$fieldName] = ['prevalue' => $row['prevalue'], 'postvalue' => $row['postvalue']];
			$rows[] = $row;
		}
		$dataReader->close();
		return $rows;
	}

	/**
	 * {@inheritDoc}
	 */
	public function setReviewed(int|string $id, array $lastReviewedUsers)
	{
		return $this->db->createCommand()->update(
			'vtiger_modtracker_basic',
			['last_reviewed_users' => '#' . implode('#', array_filter($lastReviewedUsers)) . '#'],
			['id' => (int) $id]
		)->execute();
	}

	/**
	 * {@inheritDoc}
	 */
	public function getReviewChanges(int $crmId, int $status)
	{
		$query = (new Query())
			->select(['last_reviewed_users', 'id', 'changedon'])
			->from('vtiger_modtracker_basic')
			->where(['crmid' => $crmId])
			->andWhere(['<>', 'status', $status])
			->orderBy(['changedon' => SORT_DESC, 'id' => SORT_DESC]);

		$dataReader = $query->createCommand($this->db)->query();
		while ($row = $dataReader->read()) {
			$row['last_reviewed_users'] = array_filter(explode('#', $row['last_reviewed_users']));
			yield $row;
		}
		$dataReader->close();
	}

	/**
	 * {@inheritDoc}
	 */
	public function getDetail(int|string $id): array
	{
		$dataReader = (new Query())->from('vtiger_modtracker_detail')->where(['id' => (int) $id])->createCommand()->query();
		$fields = [];
		while ($row = $dataReader->read()) {
			$fields[$row['fieldname']] = ['prevalue' => $row['prevalue'], 'postvalue' => $row['postvalue']];
		}
		$dataReader->close();

		return $fields;
	}

	/**
	 * {@inheritDoc}
	 */
	public function getInventoryChanges(int|string $id): array
	{
		$changes = (new Query())->select(['changes'])->from('u_#__modtracker_inv')->where(['id' => (int) $id])->scalar();

		return $changes ? \App\Json::decode($changes) : [];
	}

	/**
	 * {@inheritDoc}
	 */
	public function changeEntryOwner(int $oldOwnerId, int $newOwnerId): void
	{
		$this->db->createCommand()->update('vtiger_modtracker_basic', ['whodid' => $newOwnerId], ['whodid' => $oldOwnerId])->execute();
	}

	/**
	 * {@inheritDoc}
	 */
	public function getRecentActivityForRecord(int $recordId, string $startDateTime, array $statuses, int $authorIdExclusion, int $limit = 5): array
	{
		$query = (new Query())
			->from('vtiger_modtracker_basic')
			->where(['vtiger_modtracker_basic.crmid' => $recordId])
			->andWhere(['>=', 'vtiger_modtracker_basic.changedon', $startDateTime])
			->andWhere(['<>', 'vtiger_modtracker_basic.whodid', $authorIdExclusion])
			->andWhere(['vtiger_modtracker_basic.status' => $statuses])
			->orderBy(['vtiger_modtracker_basic.id' => SORT_ASC])->limit($limit);

		return $query->createCommand()->query()->readAll();
	}

	/**
	 * {@inheritDoc}
	 */
	public function getActivityForRecord(int $recordId, array $statusesExclusion, int $limit, int $offset, ?string $startingIndex = null): array
	{
		$query = (new Query())
			->from('vtiger_modtracker_basic')
			->where(['crmid' => $recordId])
			->andWhere(['not in', 'status', $statusesExclusion])
			->limit($limit)
			->offset($offset)
			->orderBy(['changedon' => SORT_DESC]);
		if ($startingIndex) {
			$query->andWhere(['>=', 'id', (int) $startingIndex]);
		}

		return $query->createCommand()->query()->readAll();
	}

	/**
	 * {@inheritDoc}
	 */
	public function getHistory(int $limit, int $offset, ?array $modulesExclusion = null, ?array $modulesInclusion = null)
	{
		$query = new Query();
		$query->select(['vtiger_modtracker_basic.*'])
			->from('vtiger_modtracker_basic')
			->innerJoin('vtiger_crmentity', 'vtiger_modtracker_basic.crmid = vtiger_crmentity.crmid')
			->where(['vtiger_crmentity.deleted' => 0]);

		if ($modulesExclusion) {
			$query->andWhere(['not in', 'module', $modulesExclusion]);
		}
		if ($modulesInclusion) {
			$query->andWhere(['in', 'module', $modulesInclusion]);
		}
		$query->orderBy(['vtiger_modtracker_basic.id' => SORT_DESC])
			->limit($limit)
			->offset($offset);

		return $query->createCommand()->query()->readAll();
	}

	/**
	 * {@inheritDoc}
	 */
	public function getUnreviewedChangesCount($recordId, array $statuses, int $userId, bool $sort): array
	{
		$query = (new Query())->select(['crmid', 'u' => 'last_reviewed_users'])->from('vtiger_modtracker_basic')
			->where(['crmid' => $recordId])
			->andWhere(['not in', 'status', $statuses]);
		if ($sort) {
			$query->addSelect(['vtiger_ossmailview.type'])
				->leftJoin('vtiger_modtracker_relations', 'vtiger_modtracker_basic.id = vtiger_modtracker_relations.id')
				->leftJoin('vtiger_ossmailview', 'vtiger_modtracker_relations.targetid = vtiger_ossmailview.ossmailviewid')
				->orderBy('vtiger_modtracker_basic.crmid ,vtiger_modtracker_basic.id DESC');
		}
		$dataReader = $query->createCommand()->query();
		$changes = [];
		while ($row = $dataReader->read()) {
			$changes[$row['crmid']][] = $row;
		}
		$dataReader->close();
		$unreviewed = [];
		foreach ($changes as $crmId => $rows) {
			$all = $mails = 0;
			foreach ($rows as $row) {
				if (str_contains($row['u'], "#$userId#")) {
					break;
				}
				if (isset($row['type']) && 1 === (int) $row['type']) {
					++$mails;
				} elseif (!isset($row['type'])) {
					++$all;
				}
			}
			$unreviewed[$crmId]['a'] = $all;
			$unreviewed[$crmId]['m'] = $mails;
		}
		return $unreviewed;
	}

	/**
	 * {@inheritDoc}
	 */
	public function getUpdatesForWidget(string $moduleName, array $actions, array $dateRange, ?int $owner, ?int $historyOwner, \Vtiger_Paging_Model $pagingModel): array
	{
		$updates = [];
		$queryGenerator = $this->getQueryGenerator($moduleName, $actions, $dateRange, $owner, $historyOwner, $pagingModel);
		$queryGenerator->setCustomColumn('vtiger_modtracker_basic.*');
		$dataReader = $queryGenerator->createQuery()->orderBy(['vtiger_modtracker_basic.id' => SORT_DESC])->createCommand()->query();
		while ($row = $dataReader->read()) {
			if (\count($updates) === $pagingModel->getPageLimit()) {
				$pagingModel->set('nextPageExists', true);
				break;
			}
			$recordModel = new \ModTracker_Record_Model();
			$recordModel->setData($row)->setParent($row['crmid'], $moduleName);
			$updates[$recordModel->getId()] = $recordModel;
		}
		return $updates;
	}

	/**
	 * {@inheritDoc}
	 */
	public function getSummary(array $modules, array $actions, array $dateRange, ?int $owner, ?int $historyOwner): array
	{
		$updates = $usedActions = [];
		$query = (new \App\Db\Query())
			->select(['vtiger_modtracker_basic.module'])
			->from('vtiger_modtracker_basic')
			->where(['and',
				['vtiger_modtracker_basic.module' => $modules],
				['vtiger_modtracker_basic.status' => $actions],
				[
					'between',
					'vtiger_modtracker_basic.changedon',
					$dateRange[0] . ' 00:00:00', $dateRange[1] . ' 23:59:59'
				]
			]);
		if ($historyOwner) {
			$query->andWhere(['vtiger_modtracker_basic.whodid' => $historyOwner]);
		}
		if ($owner) {
			$query->innerJoin('vtiger_crmentity', 'vtiger_modtracker_basic.crmid=vtiger_crmentity.crmid')
				->andWhere(['or',
					['vtiger_crmentity.smownerid' => $owner],
					['vtiger_crmentity.crmid' => (new \App\Db\Query())->select(['crmid'])->from('u_#__crmentity_showners')->where(['userid' => $owner])]
				]);
		}
		$query->distinct(true);
		$dataReader = $query->createCommand()->query();
		while ($row = $dataReader->read()) {
			$moduleName = $row['module'];
			$queryGenerator = self::getQueryGenerator($moduleName, $actions, $dateRange, $owner, $historyOwner);
			$queryGenerator->setCustomColumn(['vtiger_modtracker_basic.status', 'counter' => new \yii\db\Expression('COUNT(*)')])->setCustomGroup(['vtiger_modtracker_basic.status']);
			$result = $queryGenerator->createQuery()->createCommand()->queryAllByGroup(0);
			if ($result) {
				$updates[$moduleName] = $result;
				$usedActions = array_unique(array_merge($usedActions, array_keys($result)));
			}
		}
		$dataReader->close();
		return [$updates, $usedActions];
	}

	/**
	 * Returns statuses query part for given entry type.
	 *
	 * @param string $type
	 *
	 * @return array|array[]
	 */
	private function getConditionByType(string $type): array
	{
		return match ($type) {
			'changes' => ['not in', 'status', [\ModTracker_Record_Model::DISPLAYED, \ModTracker_Record_Model::SHOW_HIDDEN_DATA]],
			'review' => ['status' => [\ModTracker_Record_Model::DISPLAYED, \ModTracker_Record_Model::SHOW_HIDDEN_DATA]],
			default => [],
		};
	}

	/**
	 * Gets QueryGenerator object.
	 *
	 * @param string                    $moduleName
	 * @param array                     $actions
	 * @param array                     $dateRange
	 * @param int|null                  $owner
	 * @param int|null                  $historyOwner
	 * @param \Vtiger_Paging_Model|null $pagingModel
	 *
	 * @return \App\QueryGenerator
	 */
	private function getQueryGenerator(string $moduleName, array $actions, array $dateRange, ?int $owner, ?int $historyOwner, ?\Vtiger_Paging_Model $pagingModel = null): \App\QueryGenerator
	{
		$queryGenerator = (new \App\QueryGenerator($moduleName))
			->setFields([])
			->addJoin([
				'INNER JOIN',
				'vtiger_modtracker_basic',
				'vtiger_crmentity.crmid = vtiger_modtracker_basic.crmid'
			])
			->addNativeCondition(['vtiger_modtracker_basic.status' => $actions])
			->addNativeCondition([
				'between',
				'vtiger_modtracker_basic.changedon',
				$dateRange[0] . ' 00:00:00', $dateRange[1] . ' 23:59:59'
			])->setStateCondition('All');
		if ($pagingModel) {
			$queryGenerator->setLimit($pagingModel->getPageLimit() + 1)->setOffset($pagingModel->getStartIndex());
		}
		if ($owner) {
			$queryGenerator->addCondition('assigned_user_id', $owner, 'e', false);
			if ($queryGenerator->getModuleField('shownerid')) {
				$queryGenerator->addCondition('shownerid', $owner, 'e', false);
			}
		}
		if ($historyOwner) {
			$queryGenerator->addNativeCondition(['vtiger_modtracker_basic.whodid' => $historyOwner]);
		}
		return $queryGenerator;
	}
}
