<?php
/**
 * Service for process formula expression
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Łukasz Krawczyk <l.krawczyk@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Report\Decorator\ReportData;

use App\Report\Attribute\ServiceAggregator;
use App\Report\Decorator\ReportDataDecorator;
use App\Report\Enums\FunctionType;
use App\Report\Formula\FormulaProcessor;
use App\Report\Mapper\FormulaDataMapper;
use App\Report\Model\Query;
use RuntimeException;

/** FormulaReportDataDecorator class */
final class FormulaReportDataDecorator implements ReportDataDecorator
{
	/** @param iterable<FormulaProcessor> $processors */
	public function __construct(
		#[ServiceAggregator(FormulaProcessor::class)]
		private readonly iterable $processors,
		private readonly FormulaDataMapper $mapper,
	) {
	}

	/** {@inheritDoc} */
	public function decorate(array $data, Query $query): array
	{
		foreach ($query->getFormulaExpressions() as $expression) {
			$processor = $this->getProcessor($function = $expression->getFunction());

			$data[$expression->getAlias()] = $processor->process(
				$this->mapper->map($expression, $data),
				$function,
			);
		}

		return $data;
	}

	/** Get processor for formula */
	private function getProcessor(FunctionType $function): FormulaProcessor
	{
		foreach ($this->processors as $processor) {
			if ($processor->supports($function)) {
				return $processor;
			}
		}

		throw new RuntimeException(
			sprintf('No decorator found for function: %s', $function->value),
		);
	}
}
