<?php

/**
 * Record Class for ReportTemplate.
 *
 * @package   Model
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Klaudia Łozowska <k.Lozowska@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

declare(strict_types=1);

use App\Db;
use App\Db\Query;
use App\Exceptions\AppException;
use App\Module;
use App\Privilege;
use App\Report\Builder\QueryBuilder;
use App\Report\Mapper\FilterMapper;
use App\Report\Model\Template;
use App\Report\Report;
use App\Report\Repository\TemplateRepository;
use App\Report\Sanitizer\FilterSanitizer;
use App\Request;
use App\User;
use yii\db\Exception;

final class ReportTemplate_Record_Model extends Vtiger_Record_Model
{
	/** @var array[] Fields restrictions for save action. */
	private const FIELDS = [
		self::QUERY_TABLE => [
			'input' => ['base_module_id', 'expressions', 'joins', 'filter_group'],
			'db' => ['base_module_id', 'parent_id', 'data_sources'],
		],
		self::QUERY_JOIN_TABLE => [
			'input' => ['relation_id', 'type'],
			'db' => ['query_id', 'relation_id', 'type'],
		],
		self::QUERY_EXPRESSION_TABLE => [
			'input' => ['field', 'function', 'group_by', 'order_by'],
			'db' => ['query_id', 'field', 'function', 'group_by', 'order_by', 'sort', 'formula_field'],
		],
		self::QUERY_FILTER_GROUP_TABLE => [
			'input' => ['condition', 'filters', 'filter_groups'],
			'db' => ['query_id', 'condition', 'parent_id'],
		],
		self::QUERY_FILTER_TABLE => [
			'input' => ['field', 'operator', 'value', 'subquery'],
			'db' => ['filter_group_id', 'field', 'operator', 'value', 'subquery_id'],
		],
	];

	/** @var string Report template query table name. */
	private const QUERY_TABLE = 'u_yf_report_template_query';

	/** @var string Report template query join table name. */
	private const QUERY_JOIN_TABLE = 'u_yf_report_template_query_join';

	/** @var string Report template query expression table name. */
	private const QUERY_EXPRESSION_TABLE = 'u_yf_report_template_query_expression';

	/** @var string Report template query filter group table name. */
	private const QUERY_FILTER_GROUP_TABLE = 'u_yf_report_template_query_filter_group';

	/** @var string Report template query filter table name. */
	private const QUERY_FILTER_TABLE = 'u_yf_report_template_query_filter';

	/**
	 * Array to store new report template query data.
	 *
	 * @var array
	 */
	private array $newQueryData = [];

	private static ?Template $domainTemplate = null;

	/**
	 * {@inheritdoc}
	 *
	 * @throws AppException
	 * @throws Exception
	 */
	public function saveToDb(): void
	{
		parent::saveToDb();
		if ($this->newQueryData) {
			$this->resetQuery();
		}
	}

	/** {@inheritdoc} */
	public function isEditable(): bool
	{
		return parent::isEditable() && $this->loggedUserIsReportOwner();
	}

	/**
	 * Store new query from request data.
	 *
	 * @param Request $request
	 *
	 * @return void
	 */
	public function setQueryDataFromRequest(Request $request): void
	{
		$query = $request->get('query');
		$moduleName = $query['base_module_id'];
		$query['base_module_id'] = Module::getModuleId($moduleName);
		$query['filter_group'] = Report::get(FilterSanitizer::class)->sanitize(
			Report::get(QueryBuilder::class)->query($moduleName),
			$query['filter_group']
		);

		$this->newQueryData = $query;
	}

	/**
	 * {@inheritdoc}
	 */
	public function isMandatorySave(): bool
	{
		return true;
	}

	public function getRecordListViewLinksRightSide(): array
	{
		$links = $recordLinks = [];

		if (Privilege::isPermitted('Documents', 'CreateView')) {
			$recordLinks[] = [
				'linktype' => 'LIST_VIEW_ACTIONS_RECORD_RIGHT_SIDE',
				'linklabel' => 'BTN_GENERATE_REPORT',
				'linkurl' => \sprintf('javascript:ReportTemplate_List_Js.generateReport(%d)', $this->getId()),
				'linkicon' => 'fas fa-download',
				'linkclass' => 'btn-sm btn-primary',
			];
		}

		foreach ($recordLinks as $recordLink) {
			$links[] = Vtiger_Link_Model::getInstanceFromValues($recordLink);
		}

		return $links;
	}

	public function getReportFieldModel(string $expression)
	{
		[$fieldName, $fieldModuleName, $sourceFieldName] = array_pad(explode(':', $expression), 3, false);
		if (false === $fieldModuleName) {
			return Vtiger_Module_Model::getInstance($this->get('query')['base_module_name'])->getFieldByName($fieldName);
		}
		if ('INVENTORY' === $sourceFieldName) {
			return	Vtiger_Inventory_Model::getInstance($fieldModuleName)->getField($fieldName);
		}
		return Vtiger_Field_Model::getInstance($fieldName, Vtiger_Module_Model::getInstance($fieldModuleName));
	}

	/**
	 * {@inheritdoc}
	 */
	protected function loadData(int $recordId): void
	{
		parent::loadData($recordId);

		if (!$this->get('query')) {
			self::$domainTemplate = Report::get(TemplateRepository::class)->find($recordId);
			if (null !== self::$domainTemplate) {
				$query = self::prepareQueryData(self::$domainTemplate->getQuery()->getId());
				$this->set('query', $query);
				$this->set('conditions', $query['filters']);
			}
		}
	}

	private function loggedUserIsReportOwner(): bool
	{
		return $this->get('assigned_user_id') === User::getCurrentUserId();
	}

	/**
	 * Reset report template query.
	 *
	 * @throws Exception|AppException
	 *
	 * @return void
	 */
	private function resetQuery(): void
	{
		$db = Db::getInstance();
		$createCommand = $db->createCommand();
		if ($this->getId()) {
			$createCommand->delete(
				self::QUERY_TABLE,
				['id' => $this->get('query') ? $this->get('query')['id'] : $this->get('temp_id')],
			)->execute();
		}

		$queryId = $this->createQuery($this->newQueryData);

		$createCommand->update(
			$this->getEntity()->table_name,
			['query_id' => $queryId],
			['id' => $this->getId()],
		)->execute();
	}

	/**
	 * Create report template query.
	 *
	 * @param array    $data
	 * @param int|null $parentId
	 *
	 * @throws AppException
	 * @throws Exception
	 *
	 * @return int
	 */
	private function createQuery(array $data, ?int $parentId = null): int
	{
		$queryId = $this->saveItem([...$data, 'parent_id' => $parentId], self::QUERY_TABLE);

		foreach ($data['expressions'] as $item) {
			$this->saveItem([...$item, 'query_id' => $queryId], self::QUERY_EXPRESSION_TABLE);
		}

		foreach ($data['joins'] as $item) {
			$this->saveItem([...$item, 'query_id' => $queryId], self::QUERY_JOIN_TABLE);
		}
		if (!empty($data['filter_group'])) {
			$this->createFilterGroup($data['filter_group'], $queryId);
		}

		return $queryId;
	}

	/**
	 * Create report template query filter group.
	 *
	 * @param array $data
	 * @param int   $queryId
	 * @param ?int  $parentId
	 *
	 * @throws AppException
	 * @throws Exception
	 *
	 * @return void
	 */
	private function createFilterGroup(array $data, int $queryId, ?int $parentId = null): void
	{
		$filterGroupId = $this->saveItem(
			[...$data, 'query_id' => $queryId, 'parent_id' => $parentId],
			self::QUERY_FILTER_GROUP_TABLE,
		);

		foreach ($data['rules'] as $filter) {
			if (true === \array_key_exists('condition', $filter)) {
				$this->createFilterGroup($filter, $queryId, $filterGroupId);
				continue;
			}

			if (\is_array($value = $filter['value'])) {
				$filter['value'] = json_encode($value);
			}

			$this->saveItem(
				[...$filter, 'filter_group_id' => $filterGroupId],
				self::QUERY_FILTER_TABLE,
			);
		}
	}

	/**
	 * Save item to database.
	 *
	 * @param array  $data
	 * @param string $tableName
	 *
	 * @throws AppException
	 * @throws Exception
	 *
	 * @return int
	 */
	private function saveItem(array $data, string $tableName): int
	{
		$this->validateInputFields($data, $tableName);

		$db = Db::getInstance();
		$createCommand = $db->createCommand();

		$data = array_intersect_key($data, array_flip(self::FIELDS[$tableName]['db']));

		$createCommand->insert($tableName, $data)->execute();

		return (int) $db->getLastInsertID($tableName);
	}

	/**
	 * Validate input fields.
	 *
	 * @param array  $data
	 * @param string $tableName
	 *
	 * @throws AppException
	 *
	 * @return void
	 */
	private function validateInputFields(array $data, string $tableName): void
	{
		$context = match ($tableName) {
			self::QUERY_TABLE => 'query',
			self::QUERY_JOIN_TABLE => 'join',
			self::QUERY_EXPRESSION_TABLE => 'expression',
			self::QUERY_FILTER_GROUP_TABLE => 'filter_group',
			self::QUERY_FILTER_TABLE => 'filter',
			default => '',
		};

		foreach (self::FIELDS[$tableName]['input'] as $field) {
			if (!\array_key_exists($field, $data)) {
				//	throw new AppException(sprintf('Missing value: %s:%s', $context, $field));
			}
		}
	}

	/**
	 * Get report template query data.
	 *
	 * @param int $queryId
	 *
	 * @return array
	 */
	private static function prepareQueryData(int $queryId): array
	{
		$data = (new Query())
			->from('u_yf_report_template_query')
			->where(['id' => $queryId, 'parent_id' => null])
			->one();

		return [
			'id' => $data['id'],
			'base_module_id' => $data['base_module_id'],
			'data_sources' => $data['data_sources'],
			'base_module_name' => Module::getModuleName($data['base_module_id']),
			'expressions' => self::prepareQueryExpressionsData($queryId),
			'joins' => self::prepareQueryJoinsData($queryId),
			'filters' => self::prepareQueryFiltersData(),
		];
	}

	/**
	 * Get report template query `join` data.
	 *
	 * @param int $queryId
	 *
	 * @return array
	 */
	private static function prepareQueryJoinsData(int $queryId): array
	{
		$dataReader = (new Query())
			->from('u_yf_report_template_query_join')
			->where(['query_id' => $queryId])
			->createCommand()
			->query();

		$data = [];
		while ($row = $dataReader->read()) {
			$data[$row['id']]['relation_id'] = $row['relation_id'];
			$data[$row['id']]['type'] = $row['type'];
		}

		return $data;
	}

	/**
	 * Get report template query `expression` data.
	 *
	 * @param int $queryId
	 *
	 * @return array
	 */
	private static function prepareQueryExpressionsData(int $queryId): array
	{
		$report = Report::get(TemplateRepository::class);
		return $report->findExpressionsByQueryId($queryId);
	}

	/**
	 * Get report template query `filter` data.
	 *
	 * @return array
	 */
	private static function prepareQueryFiltersData(): array
	{
		$filterGroup = self::$domainTemplate->getQuery()->getFilterGroup();
		return null !== $filterGroup ? Report::get(FilterMapper::class)->map($filterGroup) : [];
	}
}
