<?php

/**
 * OAuth authorization action file.
 *
 * @package Action
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

use App\Config;
use App\Controller\Action;
use App\Exceptions\NoPermittedToRecord;
use App\Mail\Account;
use App\Mail\Account\Repository;
use App\Mail\Account\Repository\AccountPrivateRepository;
use App\Request;
use App\Session;
use App\User;

/**
 * OAuth authorization action class.
 */
class Users_OAuth_Action extends Action
{
	/** {@inheritdoc} */
	public function checkPermission(Request $request)
	{
		$recordId = $request->getInteger('record');
		$baseId = User::getCurrentUserRealId();
		$currentUserId = User::getCurrentUserId();
		$repository = new AccountPrivateRepository();
		$accountEntity = $repository->findByUserId($recordId);

		if ($currentUserId !== $recordId || !$accountEntity || !$accountEntity->isPrivate() || ($baseId !== $currentUserId && !User::getUserModel($baseId)->isAdmin())) {
			throw new NoPermittedToRecord('ERR_NO_PERMISSIONS_FOR_THE_RECORD', 406);
		}
	}

	/** {@inheritdoc} */
	public function process(Request $request)
	{
		$recordId = $request->getInteger('record');
		$class = AccountPrivateRepository::class;

		$repository = (new Repository())->get($class);
		$accountEntity = $repository->findByUserId($recordId);
		$mailAccount = new Account($accountEntity, $repository);
		$oauthProvider = $mailAccount->getOAuthProvider();

		$state = $oauthProvider->getState();
		$options = ['login_hint' => $mailAccount->getLogin()];
		if (!$accountEntity->getRefreshToken() || !$accountEntity->getPassword()) {
			$options['prompt'] = 'consent';
		}
		$url = $oauthProvider->getAuthorizationUrl($options);

		$hash = sha1($state);
		Session::set("OAuth.State.{$hash}", [
			'state' => $state,
			'recordId' => $accountEntity->getId(),
			'redirectUri' => Config::main('site_URL') . "index.php?module=Users&view=PreferenceDetail&record={$recordId}&tab=mailbox",
			'repositoryClass' => $class
		]);

		header('location: ' . $url);
		exit;
	}

	/** {@inheritdoc} */
	public function validateRequest(Request $request)
	{
		$request->validateReadAccess();
	}
}
