/* {[The file is published on the basis of YetiForce Public License 7.0 that can be found in the following directory: licenses/LicenseEN.txt or yetiforce.com]} */
'use strict';

/**
 * Mass Edit Modal JS class for Vtiger modules.
 */
$.Class(
	'Base_MassEditModal_JS',
	{
		/** @property {string} massEditPreSave Event name for pre-save hook */
		massEditPreSave: 'Vtiger.MassEdit.PreSave'
	},
	{
		/**
		 * Registers event for tab click to validate form fields before switching tabs.
		 * @returns {void}
		 */
		registerEventForTabClick: function () {
			this.form.find('.massEditTabs').on('click', 'a[data-toggle="tab"]', (e) => {
				this.form.validationEngine('validate');
				if (this.form.data('jqv').InvalidFields.length > 0) {
					e.stopPropagation();
				}
			});
		},

		/**
		 * Handles field validation activation/deactivation and related reference fields on row selection.
		 * @returns {void}
		 */
		postMassEdit: function () {
			const editInstance = Vtiger_Edit_Js.getInstance();
			this.container.find('.selectRow').on('change', (e) => {
				let element = $(e.currentTarget);
				let blockElement = element.closest('.js-form-row-container').find('.fieldValue');
				let fieldElement = blockElement.find('[data-validation-engine],[data-invalid-validation-engine]');
				let fieldInfo = fieldElement.data('fieldinfo');
				if (element.prop('checked')) {
					this.activeFieldValidation(fieldElement);
				} else {
					this.inactiveFieldValidation(fieldElement);
				}
				if (fieldInfo !== undefined && fieldInfo.type === 'reference') {
					let mapFields = editInstance.getMappingRelatedField(
						fieldInfo.name,
						editInstance.getReferencedModuleName(blockElement),
						this.container
					);
					$.each(mapFields, function (key, _value) {
						let checkboxElement = this.container.find('[id="selectRow' + key + '"]');
						if (checkboxElement.length && checkboxElement.prop('disabled')) {
							checkboxElement.prop('disabled', false);
							checkboxElement.trigger('click');
							checkboxElement.prop('disabled', true);
						}
					});
				}
			});
			this.container.find('form').on('submit', (e) => {
				let form = $(e.currentTarget);
				if (typeof form.data('submit') !== 'undefined') {
					return false;
				}
				if (form.validationEngine('validate')) {
					e.preventDefault();
					if (!form.find('input[id^="selectRow"]:checked').length) {
						app.showNotify({
							text: app.vtranslate('JS_NONE_FIELD_MARKED_IN_MASS_EDIT'),
							type: 'error'
						});
						return;
					}
					let invalidFields = form.data('jqv').InvalidFields;
					if (invalidFields.length == 0) {
						form.find('[name="saveButton"]').prop('disabled', true);
					} else {
						return;
					}
					this.saveForm();
				} else {
					form.removeData('submit');
					app.formAlignmentAfterValidation(form);
				}
			});
		},

		/**
		 * Activates validation for a field.
		 * @param {jQuery} field
		 * @returns {void}
		 */
		activeFieldValidation: function (field) {
			let validationVal = field.attr('data-invalid-validation-engine');
			if (typeof validationVal === 'undefined') return;
			field.attr('data-validation-engine', validationVal);
			field.removeAttr('data-invalid-validation-engine');
		},

		/**
		 * Deactivates validation for a field and removes it from invalid fields.
		 * @param {jQuery} field
		 * @returns {void}
		 */
		inactiveFieldValidation: function (field) {
			field.validationEngine('hide');
			const invalidFields = this.form.data('jqv').InvalidFields;
			let fields = [field.get(0)];
			field.attr('data-invalid-validation-engine', field.attr('data-validation-engine'));
			field.removeAttr('data-validation-engine');

			if (field.is('select') && field.hasClass('select2')) {
				const selectElement = app.getSelect2ElementFromSelect(field);
				selectElement.validationEngine('hide');
				fields.push(selectElement.get(0));
			}
			for (let i in fields) {
				let response = $.inArray(fields[i], invalidFields);
				if (response != '-1') {
					invalidFields.splice(response, 1);
				}
			}
		},

		/**
		 * Saves the mass edit form after validation and triggers the mass action save.
		 * @returns {void}
		 */
		saveForm: function () {
			let massEditPreSaveEvent = $.Event(Base_MassEditModal_JS.massEditPreSave);
			this.form.trigger(massEditPreSaveEvent);
			if (massEditPreSaveEvent.isDefaultPrevented()) {
				this.form.find('[name="saveButton"]').removeAttr('disabled');
				return;
			}
			this.form.find('[id^="selectRow"]').each(function (_index, checkbox) {
				checkbox = $(checkbox);
				if (!checkbox.prop('checked')) {
					checkbox
						.closest('.js-form-row-container')
						.find('.fieldValue [name]')
						.each(function (_index, element) {
							element = $(element);
							element.attr('data-element-name', element.attr('name')).removeAttr('name');
						});
				}
			});
			switch (this.container.data('sourceView')) {
				case 'List':
					const listInstance = Vtiger_List_Js.getInstance();
					listInstance
						.massActionSave(this.form)
						.done(() => {
							listInstance.getListViewRecords();
							Vtiger_List_Js.clearList();
						})
						.fail(function (error, err) {
							app.errorLog(error, err);
						});

					break;
				case 'RelationList':
					this.baseWindow.Vtiger_RelatedList_Js.relatedListInstance
						.massActionSave(this.form)
						.done(() => {
							this.baseWindow.Vtiger_Detail_Js.reloadRelatedList();
						})
						.fail(function (error, err) {
							app.errorLog(error, err);
						});
					break;
				default:
					break;
			}
		},

		/**
		 * Deactivates validation for all fields in the form.
		 * This will remove data-validation-engine attr of all the elements.
		 * @returns {void}
		 */
		inactiveFieldsValidation: function () {
			const form = this.form;
			form.find('.fieldValue [data-validation-engine][data-fieldinfo]').each((_, e) => {
				let fieldElement = $(e);
				let fieldInfo = fieldElement.data('fieldinfo');
				if (fieldInfo.type == 'reference') {
					// get the element which will be shown which has "_display" appended to actual field name
					fieldElement = form.find('[name="' + fieldInfo.name + '_display"]');
				} else if (fieldInfo.type == 'multipicklist' || fieldInfo.type == 'sharedOwner') {
					fieldElement = form.find('[name="' + fieldInfo.name + '[]"]');
				}
				// Not all the fields will be enabled for mass edit
				if (fieldElement.length == 0) {
					return;
				}
				let elemData = fieldElement.data();
				// Blank validation by default
				let validationVal = 'validate[]';
				if ('validationEngine' in elemData) {
					validationVal = elemData.validationEngine;
					delete elemData.validationEngine;
				}
				fieldElement.attr('data-invalid-validation-engine', validationVal);
				fieldElement.removeAttr('data-validation-engine');
			});
		},

		/**
		 * Registers all necessary events and initializes components for the mass edit modal.
		 *
		 * @param {jQuery} container - The jQuery object representing the modal container element.
		 * @returns {void}
		 */
		registerEvents(container) {
			this.container = container;
			this.form = container.find('form');
			this.form.validationEngine(app.validationEngineOptions);
			this.registerEventForTabClick();
			Vtiger_Edit_Js.getInstance().registerBasicEvents(this.form);
			this.postMassEdit();
			app.showScrollBar($('div[name="massEditContent"]'), {
				height: app.getScreenHeight(70) + 'px'
			});
			this.inactiveFieldsValidation();
		}
	}
);
