<?php

/**
 * Comarch taxes synchronization file.
 *
 * The file is part of the paid functionality. Using the file is allowed only after purchasing a subscription.
 * File modification allowed only with the consent of the system producer.
 *
 * @package Integration
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */

namespace App\Integrations\Comarch\Xl\Synchronizer;

/**
 * Comarch taxes synchronization class.
 */
class Taxes extends \App\Integrations\Comarch\Synchronizer
{
	/** @var array Cache for data from the API */
	private $cache;
	/** @var array ID by name cache from the API */
	private $cacheList = [];
	/** @var \Settings_Inventory_Record_Model[] ID by name cache from the API */
	private $globalTaxes = [];

	/** {@inheritdoc} */
	public function process(): void
	{
		$this->globalTaxes = \Settings_Inventory_Record_Model::getDataAll('Taxes');
		$this->getAllFromApi();
		$this->import();
	}

	/**
	 * Import account type from API.
	 *
	 * @return void
	 */
	public function import(): void
	{
		$this->processStart('import', false);
		$i = 0;
		foreach ($this->cache as $taxName => $taxValue) {
			if ($taxModel = $this->findTax($taxName, $taxValue)) {
				if (1 == $taxModel->get('status')) {
					$taxModel->set('status', 0);
					$taxModel->save();
					++$i;
				}
			} else {
				$taxModel = new \Settings_Inventory_Record_Model();
				$taxModel->setType('Taxes');
				$taxModel->set('name', $taxName);
				$taxModel->set('value', $taxValue);
				$taxModel->set('status', 0);
				$taxModel->set('default', 0);
				$taxModel->save();
				++$i;
			}
			$this->cacheList[$taxModel->getId()] = $taxName;
		}
		$this->processEnd('import', ['imported' => $i]);
	}

	/** {@inheritdoc} */
	public function getYfValue($apiValue, array $field)
	{
		$this->loadCacheList();
		$key = array_search($apiValue, $this->cacheList);
		if (null !== $key) {
			return [
				'id' => $key,
				'value' => $this->cache[$apiValue],
			];
		}
		return null;
	}

	/** {@inheritdoc} */
	public function getApiValue($yfValue, array $field)
	{
		$this->loadCacheList();
		$yfValue = \App\Json::decode($yfValue);
		if (empty($yfValue['globalId'])) {
			throw new \App\Exceptions\AppException('ERR_REQUIRED_VALUE_MISSING');
		}
		if (empty($this->cacheList[$yfValue['globalId']])) {
			throw new \App\Exceptions\AppException('ERR_REQUIRED_VALUE_MISSING');
		}
		return [
			'id' => $this->cacheList[$yfValue['globalId']],
			'value' => $yfValue['globalTax'],
		];
	}

	/**
	 * Load cache list.
	 *
	 * @return void
	 */
	private function loadCacheList(): void
	{
		if (empty($this->cacheList)) {
			$this->process();
		}
	}

	/**
	 * Get all account type from API.
	 *
	 * @return array
	 */
	private function getAllFromApi(): array
	{
		if (null === $this->cache) {
			$this->cache = [];
			try {
				foreach ($this->getFromApi('Dictionary/VatGroup') as $value) {
					$this->cache[$value['grupaVat']] = $value['stawkaVat'];
				}
			} catch (\Throwable $ex) {
				$this->logError('getAllFromApi ' . $this->name, null, $ex);
			}
		}
		return $this->cache;
	}

	/**
	 * Find tax in YF.
	 *
	 * @param string $taxName
	 * @param string $taxValue
	 *
	 * @return \Settings_Inventory_Record_Model|null
	 */
	private function findTax(string $taxName, string $taxValue): ?\Settings_Inventory_Record_Model
	{
		$return = null;
		foreach ($this->globalTaxes as $taxModel) {
			if ($taxModel->get('name') == $taxName && $taxModel->get('value') == $taxValue) {
				$return = $taxModel;
				break;
			}
		}
		return $return;
	}
}
