<?php
/* +***********************************************************************************
 * The contents of this file are subject to the vtiger CRM Public License Version 1.0
 * ("License"); You may not use this file except in compliance with the License
 * The Original Code is:  vtiger CRM Open Source
 * The Initial Developer of the Original Code is vtiger.
 * Portions created by vtiger are Copyright (C) vtiger.
 * All Rights Reserved.
 * Contributor(s): YetiForce S.A.
 * *********************************************************************************** */

class ModTracker_Record_Model extends Vtiger_Record_Model
{
	const UPDATE = 0;
	const DELETE = 1;
	const CREATE = 2;
	const ACTIVE = 3;
	const LINK = 4;
	const UNLINK = 5;
	const CONVERTTOACCOUNT = 6;
	const DISPLAYED = 7;
	const ARCHIVED = 8;
	const REMOVED = 9;
	const TRANSFER_EDIT = 10;
	const TRANSFER_DELETE = 11;
	const TRANSFER_UNLINK = 12;
	const TRANSFER_LINK = 13;
	const SHOW_HIDDEN_DATA = 14;

	/**
	 * Status labels.
	 *
	 * @var string[]
	 */
	public static $statusLabel = [
		0 => 'LBL_UPDATED',
		1 => 'LBL_DELETED',
		2 => 'LBL_CREATED',
		3 => 'LBL_ACTIVE',
		4 => 'LBL_ADDED',
		5 => 'LBL_UNLINK',
		6 => 'LBL_CONVERTED_FROM_LEAD',
		7 => 'LBL_DISPLAYED',
		8 => 'LBL_ARCHIVED',
		9 => 'LBL_REMOVED',
		10 => 'LBL_TRANSFER_EDIT',
		11 => 'LBL_TRANSFER_DELETE',
		12 => 'LBL_TRANSFER_UNLINK',
		13 => 'LBL_TRANSFER_LINK',
		14 => 'LBL_SHOW_HIDDEN_DATA',
	];

	/** @var \Vtiger_Record_Model Parent record model */
	protected $parent;

	private $inventoryChanges;

	/**
	 * @var \App\ModTracker\Repository\ModTrackerRepositoryInterface
	 */
	private \App\ModTracker\Repository\ModTrackerRepositoryInterface $modTrackerRepository;

	/**
	 * {@inheritDoc}
	 */
	public function __construct($values = [])
	{
		$this->modTrackerRepository = \App\ModTracker\Repository::instance();

		parent::__construct($values);
	}

	/**
	 * Function to get the history of updates on a record.
	 *
	 * @param int                 $parentRecordId
	 * @param Vtiger_Paging_Model $pagingModel
	 * @param string              $type
	 * @param int|null            $startWith
	 *
	 * @return self[] - list of  ModTracker_Record_Model
	 */
	public static function getUpdates(int $parentRecordId, Vtiger_Paging_Model $pagingModel, string $type, ?int $startWith = null)
	{
		$recordInstances = [];
		$updates = \App\ModTracker\Repository::instance()
			->getUpdates($parentRecordId, $pagingModel, $type, $startWith);
		foreach ($updates as $row) {
			$recordInstance = new self();
			$recordInstance->setData($row)->setParent($row['crmid'], $row['module']);
			$recordInstances[] = $recordInstance;
		}

		return $recordInstances;
	}

	public static function setLastReviewed($recordId)
	{
		return \App\ModTracker\Repository::instance()->setLastReviewed(
			Users_Record_Model::getCurrentUserModel()->getRealId(),
			$recordId,
			self::DISPLAYED
		);
	}

	public static function unsetReviewed($recordId, $userId = false, $exception = false)
	{
		if (!$userId) {
			$currentUser = Users_Record_Model::getCurrentUserModel();
			$userId = $currentUser->getRealId();
		}

		return \App\ModTracker\Repository::instance()
			->unsetReviewed($userId, $recordId, self::DISPLAYED, $exception);
	}

	/**
	 * Checks if is new changes.
	 *
	 * @param int $recordId
	 * @param int $userId
	 *
	 * @return bool
	 */
	public static function isNewChange(int $recordId, int $userId = 0): bool
	{
		if (0 === $userId) {
			$userId = App\User::getCurrentUserId();
		}

		$lastReviewedUsers = \App\ModTracker\Repository::instance()
			->getLastReviewedUsers($recordId, self::DISPLAYED);

		return $lastReviewedUsers ? !\in_array($userId, $lastReviewedUsers) : true;
	}

	/**
	 * Gets unreviewed entries.
	 *
	 * @param int|int[] $recordsId
	 * @param bool|int  $userId
	 * @param bool      $sort
	 *
	 * @return array
	 */
	public static function getUnreviewed($recordsId, $userId = false, $sort = false)
	{
		if (false === $userId) {
			$userId = \App\User::getCurrentUserId();
		}

		return \App\ModTracker\Repository::instance()
			->getUnreviewedChangesCount($recordsId, [self::DISPLAYED, self::SHOW_HIDDEN_DATA], $userId, $sort);
	}

	/**
	 * Function to get the name of the module to which the record belongs.
	 *
	 * @return Vtiger_Module_Model
	 */
	public function getModule(): Vtiger_Module_Model
	{
		if (empty($this->parent)) {
			return Vtiger_Module_Model::getInstance($this->getModuleName());
		}
		return $this->getParent()->getModule();
	}

	/**
	 * Function to get the name of the module to which the record belongs.
	 *
	 * @return string - Record Module Name
	 */
	public function getModuleName(): string
	{
		return $this->get('module');
	}

	/**
	 * Function to get the Detail View url for the record.
	 *
	 * @return string - Record Detail View Url
	 */
	public function getDetailViewUrl()
	{
		$moduleName = $this->getModuleName();
		switch ($moduleName) {
			case 'Documents':
				return 'file.php?module=Documents&action=DownloadFile&record=' . $this->get('crmid');
			case 'OSSMailView':
				$action = 'view=preview';
				break;
			default:
				$action = 'view=Detail';
				break;
		}
		return "index.php?module=$moduleName&$action&record=" . $this->get('crmid');
	}

	/**
	 * Undocumented function.
	 *
	 * @param int    $id
	 * @param string $moduleName
	 *
	 * @return $this
	 */
	public function setParent($id, $moduleName)
	{
		$this->parent = Vtiger_Record_Model::getInstanceById($id, $moduleName);
		return $this;
	}

	public function getParent()
	{
		return $this->parent;
	}

	public function checkStatus($callerStatus)
	{
		$status = $this->get('status');
		if ($status == $callerStatus) {
			return true;
		}
		return false;
	}

	public function isConvertToAccount()
	{
		return $this->checkStatus(self::CONVERTTOACCOUNT);
	}

	public function isCreate()
	{
		return $this->checkStatus(self::CREATE);
	}

	public function isUpdate()
	{
		return $this->checkStatus(self::UPDATE);
	}

	public function isRelationLink()
	{
		return $this->checkStatus(self::LINK);
	}

	public function isRelationUnLink()
	{
		return $this->checkStatus(self::UNLINK);
	}

	public function isDisplayed()
	{
		return $this->checkStatus(self::DISPLAYED);
	}

	/**
	 * Function check if status is Transfer.
	 *
	 * @return bool
	 */
	public function isTransferEdit()
	{
		return $this->checkStatus(static::TRANSFER_EDIT);
	}

	/**
	 * Function check if status is Transfer.
	 *
	 * @return bool
	 */
	public function isTransferLink()
	{
		return $this->checkStatus(static::TRANSFER_LINK);
	}

	/**
	 * Function check if status is Transfer.
	 *
	 * @return bool
	 */
	public function isTransferUnLink()
	{
		return $this->checkStatus(static::TRANSFER_UNLINK);
	}

	/**
	 * Function check if status is Transfer.
	 *
	 * @return bool
	 */
	public function isTransferDelete()
	{
		return $this->checkStatus(static::TRANSFER_DELETE);
	}

	/**
	 * Function check if status is Transfer.
	 *
	 * @return bool
	 */
	public function isShowHiddenData()
	{
		return $this->checkStatus(static::SHOW_HIDDEN_DATA);
	}

	/**
	 * Has changed state.
	 *
	 * @return bool
	 */
	public function isChangeState()
	{
		return \in_array($this->get('status'), [1, 3, 8]);
	}

	public function isReviewed($userId = false)
	{
		if (false === $userId) {
			$currentUser = Users_Record_Model::getCurrentUserModel();
			$userId = $currentUser->getId();
		}
		$reviewed = $this->get('last_reviewed_users');
		if (!\is_array($reviewed)) {
			return false;
		}

		return \in_array($userId, $reviewed, true);
	}

	/**
	 * Function to get the id of the record.
	 *
	 * @return int - Record Id
	 */
	public function getId()
	{
		return (string) $this->get('id');
	}

	/**
	 * Get status label.
	 *
	 * @return string
	 */
	public function getStatusLabel()
	{
		return static::$statusLabel[$this->get('status')];
	}

	/**
	 * Get the modifier object.
	 *
	 * @return \App\User
	 */
	public function getModifiedBy()
	{
		return \App\User::getUserModel($this->get('whodid'));
	}

	/**
	 * Get name for modifier by.
	 *
	 * @return string|bool
	 */
	public function getModifierName()
	{
		return \App\Fields\Owner::getUserLabel($this->get('whodid'));
	}

	public function getDisplayActivityTime()
	{
		$time = $this->getActivityTime();
		$time = new DateTimeField($time);

		return $time->getFullcalenderDateTimevalue();
	}

	public function getActivityTime()
	{
		return $this->get('changedon');
	}

	/**
	 * {@inheritDoc}
	 */
	public function setData($values)
	{
		if (isset($values['_id'])) {
			$values['id'] = $values['_id'];
		}

		$this->value = $values;

		return $this;
	}

	/**
	 * Function return Modtracker Field Model.
	 *
	 * @return \ModTracker_Field_Model[]
	 */
	public function getFieldInstances()
	{
		$fieldInstances = [];
		if ($this->isCreate() || $this->isUpdate() || $this->isTransferEdit()) {
			if (!$this->has('detail')) {
				$this->set('detail', $this->modTrackerRepository->getDetail($this->get('id')));
			}

			foreach ($this->get('detail') as $fieldName => $meta) {
				if ('record_id' === $fieldName || 'record_module' === $fieldName) {
					continue;
				}

				if (!($fieldModel = $this->getModule()->getFieldByName($fieldName))) {
					continue;
				}

				$meta['prevalue'] = html_entity_decode((string) $meta['prevalue']);
				$meta['postvalue'] = html_entity_decode((string) $meta['postvalue']);
				$fieldInstance = new ModTracker_Field_Model();
				$fieldInstance->setData($meta)->setParent($this->getParent())->setFieldInstance($fieldModel);
				$fieldInstances[] = $fieldInstance;
			}
		}
		return $fieldInstances;
	}

	/**
	 * Gets inventory changes.
	 *
	 * @throws \App\Exceptions\AppException
	 *
	 * @return array
	 */
	public function getInventoryChanges()
	{
		if (!isset($this->inventoryChanges)) {
			$changes = [];
			if ($this->isCreate() || $this->isUpdate() || $this->isTransferEdit()) {
				$inventoryModel = Vtiger_Inventory_Model::getInstance($this->getParent()->getModuleName());
				$data = $this->modTrackerRepository->getInventoryChanges($this->getId());
				foreach ($data as $key => $changed) {
					if (!\vtlib\Functions::getCRMRecordMetadata($changed['item'])) {
						continue;
					}
					$changes[$key]['item'] = $changed['item'];
					$changes[$key]['historyState'] = empty($changed['prevalue']) ? 'LBL_INV_ADDED' : (empty($changed['postvalue']) ? 'LBL_INV_DELETED' : 'LBL_INV_UPDATED');
					$changes[$key]['data'] = [];
					foreach ($changed['prevalue'] as $fieldName => $value) {
						if ($inventoryModel->isField($fieldName)) {
							$changes[$key]['data'][$fieldName]['field'] = $inventoryModel->getField($fieldName);
							$changes[$key]['data'][$fieldName]['prevalue'] = $value;
						}
					}
					foreach ($changed['postvalue'] as $fieldName => $value) {
						if ($inventoryModel->isField($fieldName)) {
							$changes[$key]['data'][$fieldName]['field'] = $inventoryModel->getField($fieldName);
							$changes[$key]['data'][$fieldName]['postvalue'] = $value;
						}
					}
				}
			}
			$this->inventoryChanges = $changes;
		}
		return $this->inventoryChanges;
	}

	/**
	 * Function return modtracker relation model.
	 *
	 * @return \ModTracker_Relation_Model|null
	 */
	public function getRelationInstance(): ?ModTracker_Relation_Model
	{
		if ($this->isRelationLink() || $this->isRelationUnLink() || $this->isTransferLink() || $this->isTransferUnLink()) {
			$row = $this->modTrackerRepository->findRelationById($this->get('id'));
			$relationInstance = new ModTracker_Relation_Model();
			$relationInstance->setData($row)->setParent($this);
			return $relationInstance;
		}

		return null;
	}

	/**
	 * Adds Convert To Account type entry.
	 *
	 * @param $sourceModule
	 * @param $sourceId
	 * @param $current_user
	 *
	 * @return void
	 */
	public static function addConvertToAccountRelation($sourceModule, $sourceId, $current_user)
	{
		$id = \App\ModTracker\Repository::instance()
			->createEntry([
				'crmid' => $sourceId,
				'module' => $sourceModule,
				'whodid' => $current_user,
				'changedon' => date('Y-m-d H:i:s'),
				'status' => self::CONVERTTOACCOUNT,
				'last_reviewed_users' => [App\User::getCurrentUserRealId()],
			]);

		self::unsetReviewed($sourceId, \App\User::getCurrentUserRealId(), $id);
	}

	/**
	 * Function sets the closest time-wise related record from selected modules.
	 *
	 * @param int    $sourceId
	 * @param string $sourceModule
	 * @param bool   $byUser
	 *
	 * @return array|false
	 */
	public static function setLastRelation($sourceId, $sourceModule, $byUser = false)
	{
		$db = \App\Db::getInstance();
		$userId = \App\User::getCurrentUserId();
		$query = Vtiger_HistoryRelation_Widget::getQuery($sourceId, $sourceModule, Vtiger_HistoryRelation_Widget::getActions());
		if (!$query) {
			return false;
		}
		$data = $query->limit(1)->one();
		$type = $data ? $data['type'] : '';
		$where = ['crmid' => $sourceId];
		if ($byUser) {
			$where['userid'] = $userId;
		}
		$db->createCommand()->delete('u_#__timeline', $where)->execute();
		$db->createCommand()->insert('u_#__timeline', [
			'crmid' => $sourceId,
			'type' => $type,
			'userid' => $userId,
		])->execute();

		return [$sourceId => $type];
	}

	/**
	 * Function gets the closest time-wise related record from database.
	 *
	 * @param int    $sourceIds
	 * @param string $sourceModule
	 *
	 * @return array
	 */
	public static function getLastRelation($sourceIds, $sourceModule)
	{
		$colors = Vtiger_HistoryRelation_Widget::$colors;
		if (!\is_array($sourceIds)) {
			$sourceIds = [$sourceIds];
		}
		$data = (new \App\Db\Query())->from('u_#__timeline')->where(['crmid' => $sourceIds, 'userid' => \App\User::getCurrentUserId()])->createCommand()->queryAllByGroup(1);
		if (\count($data) !== \count($sourceIds)) {
			$reSearch = array_diff_key(array_flip($sourceIds), $data);
			foreach (array_keys($reSearch) as $id) {
				$result = self::setLastRelation($id, $sourceModule, true);
				if ($result) {
					$data[key($result)]['type'] = current($result);
				}
			}
		}
		foreach ($data as $id => &$type) {
			if (isset($colors[$type['type']])) {
				$type['color'] = $colors[$type['type']];
			} else {
				$type['color'] = false;
			}
			if (false !== strpos($type['type'], 'OSSMailView')) {
				$type['type'] = 'OSSMailView';
			}
		}
		return $data;
	}
}
