<?php

/**
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */

use App\Config;
use App\EmailParser;
use App\EventHandler;
use App\Fields\File;
use App\Language;
use App\ModuleHierarchy;
use App\Privilege;
use App\Purifier;
use App\QueryGenerator;
use App\Record;
use App\Request;
use App\TextParser;
use App\User;

class OSSMail_Module_Model extends Vtiger_Module_Model
{
	public function getDefaultViewName()
	{
		return 'Index';
	}

	/** {@inheritdoc} */
	public function getSettingLinks(): array
	{
		Vtiger_Loader::includeOnce('~~modules/com_vtiger_workflow/VTWorkflowUtils.php');
		$settingsLinks = [];
		if ($menu = Settings_Vtiger_MenuItem_Model::getInstance('Mail')) {
			$settingsLinks[] = [
				'linktype' => 'LISTVIEWSETTING',
				'linklabel' => 'LBL_MODULE_CONFIGURATION',
				'linkurl' => 'index.php?module=OSSMail&parent=Settings&view=Index&block=' . $menu->get('blockid') . '&fieldid=' . $menu->get('fieldid'),
				'linkicon' => 'adminIcon-mail-download-history',
			];
		}
		return $settingsLinks;
	}

	public static function getDefaultMailAccount($accounts)
	{
		return (isset($_SESSION['AutoLoginUser']) && \array_key_exists($_SESSION['AutoLoginUser'], $accounts)) ? $accounts[$_SESSION['AutoLoginUser']] : reset($accounts);
	}

	/**
	 * URL generation for internal mail clients.
	 *
	 * @param mixed $moduleName
	 * @param mixed $record
	 * @param mixed $view
	 * @param mixed $type
	 *
	 * @return string
	 */
	public static function getComposeUrl($moduleName = false, $record = false, $view = false, $type = false): string
	{
		$url = 'index.php?module=OSSMail&view=Compose';
		if ($moduleName) {
			$url .= '&crmModule=' . $moduleName;
		}
		if ($record) {
			$url .= '&crmRecord=' . $record;
		}
		if ($view) {
			$url .= '&crmView=' . $view;
		}
		if ($type) {
			$url .= '&type=' . $type;
		}
		return $url;
	}

	public static function getComposeParam(Request $request): array
	{
		$moduleName = $request->getByType('crmModule');
		$record = $request->getInteger('crmRecord');
		$type = $request->getByType('type');
		$return = [];

		if (!empty($moduleName)) {
			$return['crmmodule'] = $moduleName;
		}
		if (!empty($record)) {
			$return['crmrecord'] = $record;
		}
		if (!$request->isEmpty('crmView')) {
			$return['crmview'] = $request->getByType('crmView');
		}
		if (!$request->isEmpty('mid') && !empty($type)) {
			$return['mailId'] = (int) $request->getInteger('mid');
			$return['type'] = $type;
		}
		if (!$request->isEmpty('pdf_path')) {
			$return['filePath'] = $request->get('pdf_path');
		}
		if (!empty($moduleName)) {
			$currentUser = Users_Record_Model::getCurrentUserModel();
			$moduleConfig = Config::module($moduleName);
			if ($moduleConfig && isset($moduleConfig['SEND_IDENTITY'][$currentUser->get('roleid')])) {
				$return['from'] = $moduleConfig['SEND_IDENTITY'][$currentUser->get('roleid')];
			}
		}
		foreach (['to', 'cc', 'bcc', 'subject'] as $key) {
			if (!$request->isEmpty($key)) {
				$return[$key] = Purifier::decodeHtml($request->get($key));
			}
		}
		if (!$request->isEmpty('emails')) {
			$return['bcc'] = implode(',', $request->get('emails'));
		}

		if (('Users' === $moduleName && $record === User::getCurrentUserRealId()) || ('Users' !== $moduleName && !empty($record) && Record::isExists($record) && Privilege::isPermitted($moduleName, 'DetailView', $record))) {
			$recordModel = Vtiger_Record_Model::getInstanceById($record, $moduleName);
			$eventHandler = new EventHandler();
			$eventHandler->setRecordModel($recordModel)->setModuleName($moduleName)->setParams($return);
			$eventHandler->trigger('MailComposeParamBefore');
			$return = $eventHandler->getParams();

			$recordModel_OSSMailView = OSSMailView_Record_Model::getCleanInstance('OSSMailView');
			if ($request->isEmpty('to') && ($email = $recordModel_OSSMailView->findEmail($record, $moduleName))) {
				$return['to'] = $email;
			}
			foreach (['_to', '_cc'] as $name) {
				$content = $request->has($name) ? $request->getRaw($name) : ($return[$name] ?? '');
				if ($content) {
					$emailParser = EmailParser::getInstanceByModel($recordModel);
					$emailParser->emailoptout = false;
					$fromEmailDetails = $emailParser->setContent($content)->parse()->getContent();
					if ($fromEmailDetails) {
						$return[substr($name, -2)] = $fromEmailDetails;
					}
					if (isset($return[$name])) {
						unset($return[$name]);
					}
				}
			}
			if (!\in_array($moduleName, array_keys(array_merge(ModuleHierarchy::getModulesByLevel(0), ModuleHierarchy::getModulesByLevel(3)))) || 'Campaigns' === $moduleName) {
				$subject = '';
				if ('new' === $type || 'Campaigns' === $moduleName) {
					$return['title'] = $recordModel->getName();
					$subject .= Purifier::decodeHtml($recordModel->getName());
				}
				$recordNumber = $recordModel->getRecordNumber();
				if (!empty($recordNumber)) {
					$return['recordNumber'] = $recordNumber;
					$subject = "[$recordNumber] $subject";
				}
				if (($templateId = $request->getInteger('template', 0)) && Record::isExists($templateId, 'EmailTemplates')) {
					$params = $request->getArray('templateParams', Purifier::TEXT, [], Purifier::ALNUM);
					$templateModel = Vtiger_Record_Model::getInstanceById($templateId, 'EmailTemplates');
					$textParser = TextParser::getInstanceByModel($recordModel);
					foreach ($params as $key => $value) {
						$textParser->setParam($key, $value);
					}
					if ('Calendar' === $moduleName && !$recordModel->isEmpty('meeting_url') && !\array_key_exists('meetingUrl', $params)) {
						$textParser->setParam('meetingUrl', $recordModel->get('meeting_url'));
					}
					$subject = $textParser->setContent($templateModel->get('subject'))->parse()->getContent();
					$return['html'] = true;
					$return['body'] = $textParser->setContent($templateModel->get('content'))->parse()->getContent();
				}
				$return['subject'] = $subject;
				if ('Calendar' === $moduleName && $request->getBoolean('ics')) {
					$filePath = Config::main('tmp_dir');
					$tmpFileName = tempnam($filePath, 'ics');
					$filePath .= basename($tmpFileName);
					if (false !== file_put_contents($filePath, $recordModel->getICal())) {
						$fileName = File::sanitizeUploadFileName($recordModel->getName()) . '.ics';
						$return['filePath'] = [['path' => $filePath, 'name' => $fileName]];
					}
				}
			}

			$eventHandler->setParams($return);
			$eventHandler->trigger('MailComposeParamAfter');
			$return = $eventHandler->getParams();
		}

		return $return;
	}

	/**
	 * URL generation for external mail clients.
	 *
	 * @param mixed $moduleName
	 * @param mixed $record
	 * @param mixed $view
	 * @param mixed $type
	 *
	 * @return string
	 */
	public static function getExternalUrl($moduleName = false, $record = false, $view = false, $type = false): string
	{
		$url = 'mailto:';
		$request = new Request([]);
		if ($moduleName) {
			$request->set('crmModule', $moduleName);
		}
		if ($record) {
			$request->set('crmRecord', $record);
		}
		if ($view) {
			$request->set('crmView', $view);
		}
		if ($type) {
			$request->set('type', $type);
		}
		$param = self::getComposeParam($request);
		if (isset($param['to'])) {
			$url .= rawurlencode($param['to']);
		}
		$url .= '?';
		foreach (['cc', 'bcc'] as $value) {
			if (isset($param[$value])) {
				$url .= $value . '=' . rawurlencode($param[$value]) . '&';
			}
		}
		if (isset($param['subject'])) {
			$url .= 'subject=' . rawurlencode($param['subject']) . '&';
		}
		if (isset($param['body'])) {
			$url .= 'body=' . rawurlencode($param['body']) . '&';
		}
		return $url;
	}

	/**
	 * Get mail url from record.
	 *
	 * @param Vtiger_Record_Model $recordModel
	 * @param string              $type
	 * @param int|null            $srecord
	 * @param string|null         $smodule
	 *
	 * @return string
	 */
	public static function getExternalUrlByRecord(Vtiger_Record_Model $recordModel, string $type = 'reply', ?int $srecord = null, ?string $smodule = null): string
	{
		$data = [
			'bodyRaw' => $recordModel->get('content'),
			'subjectRaw' => $recordModel->get('subject'),
			'fromRaw' => $recordModel->get('from_email'),
			'toRaw' => $recordModel->get('to_email'),
			'ccRaw' => $recordModel->get('cc_email'),
			'date' => $recordModel->get('date'),
			'id' => $recordModel->getId(),
		];

		return static::getExternalUrlForWidget($data, $type, $srecord, $smodule);
	}

	/**
	 * Get mail url for widget.
	 *
	 * @param array  $mailData
	 * @param string $type
	 * @param int    $srecord
	 * @param string $smoduleName
	 *
	 * @return string
	 */
	public static function getExternalUrlForWidget(array $mailData, $type, $srecord = false, $smoduleName = false): string
	{
		$subject = $mailData['subjectRaw'];
		$exceptions = ['Users','OSSMailView'];
		if ($smoduleName && !in_array($smoduleName, $exceptions) && !empty($srecord) && Record::isExists($srecord, $smoduleName) && Privilege::isPermitted($smoduleName, 'DetailView', $srecord) && !\in_array($smoduleName, array_keys(array_merge(ModuleHierarchy::getModulesByLevel(0), ModuleHierarchy::getModulesByLevel(3))))) {
			$recordModel = Vtiger_Record_Model::getInstanceById($srecord, $smoduleName);
			$recordNumber = $recordModel->getRecordNumber();
			if (!empty($recordNumber) && !str_contains($subject, "[$recordNumber]")) {
				$subject = "[$recordNumber] $subject";
			}
		}

		$getSubject = static fn ($prefix) => preg_match('/^(re|fwd)\s*:/i', ltrim($raw = html_entity_decode($subject)))
			? $raw
			: ($prefix . ' ' . $raw);

		$to = (!empty($mailData['replyToRaw']) && \in_array($type, ['reply', 'replyAll']))
			? $mailData['replyToRaw']
			: $mailData['fromRaw'];
		$cc = 'replyAll' === $type ? $mailData['ccRaw'] : '';
		$body = $mailData['bodyRaw'];
		$content = '';
		$mailtoLimit = Config::component('Mail', 'MAILTO_LIMIT');

		$subject = match ($type) {
			'reply', 'replyAll' => $getSubject('Re:'),
			'forward' => $getSubject('Fwd:'),
			default => ''
		};

		if ('forward' === $type) {
			$content .= implode(PHP_EOL, [
				Language::translate('LBL_MAIL_FORWARD_INTRO', 'OSSMailView'),
				Language::translate('Subject', 'OSSMailView') . ': ' . $subject,
				Language::translate('Date', 'OSSMailView') . ': ' . $mailData['date'],
				Language::translate('From', 'OSSMailView') . ': ' . $mailData['fromRaw'],
				Language::translate('To', 'OSSMailView') . ': ' . $mailData['toRaw'],
				''
			]) . PHP_EOL;
			$to = '';
		} else {
			$content .= Language::translateArgs('LBL_MAIL_REPLY_INTRO', 'OSSMailView', $mailData['date'], $mailData['fromRaw']) . PHP_EOL;
		}

		$params = array_filter([
			'to' => $to,
			'cc' => $cc,
			'subject' => $subject,
		]);

		$config = HTMLPurifier_Config::createDefault();
		$config->set('Core.Encoding', Config::main('default_charset'));
		$config->set('Cache.SerializerPath', ROOT_DIRECTORY . '/cache/vtlib');
		$config->set('CSS.AllowTricky', false);
		$config->set('HTML.AllowedElements', 'div,p,br');
		$config->set('HTML.AllowedAttributes', '');
		$purifier = new HTMLPurifier($config);

		$body = str_replace(
			['<p> </p>', '<p></p>', '</p>', '<br />', '<p>', '<div>', '</div>', PHP_EOL . PHP_EOL],
			['', '', PHP_EOL, PHP_EOL, '', '', PHP_EOL, PHP_EOL],
			nl2br($purifier->purify($body))
		);

		$query = array_map(
			static fn ($key, $value) => rawurlencode($key) . '=' . rawurlencode($value),
			array_keys($params),
			$params
		);
		$currentLength = \strlen('mailto:?' . implode('&', $query));
		foreach (explode(PHP_EOL, $body) as $line) {
			if ($currentLength + \strlen('&body=' . rawurlencode($content . $line)) > $mailtoLimit) {
				break;
			}
			if ('' !== $line) {
				$content .= '> ' . $line . PHP_EOL;
			}
		}
		if ($content) {
			$params['body'] = $content;
		}

		$eventHandler = new EventHandler();
		if ($srecord && Record::isExists($srecord)) {
			$eventHandler->setRecordModel(Vtiger_Record_Model::getInstanceById($srecord));
		} elseif ($smoduleName) {
			$eventHandler->setModuleName($smoduleName);
		}
		$eventHandler->setParams($params);
		$eventHandler->trigger('MailComposeParamAfter');
		$params = $eventHandler->getParams();

		$mailto = 'mailto:';
		if (isset($params['to'])) {
			$mailto .= rawurldecode($params['to']);
			unset($params['to']);
		}
		if ($params) {
			$query = [];
			$currentLength = \strlen($mailto) + 1;
			foreach ($params as $key => $value) {
				$paramStr = rawurlencode($key) . '=' . rawurlencode($value);
				if ($currentLength + 1 + \strlen($paramStr) > $mailtoLimit) {
					break;
				}
				$query[] = $paramStr;
				$currentLength += 1 + \strlen($paramStr);
			}

			$mailto .= '?' . implode('&', $query);
		}

		return $mailto;
	}

	/** {@inheritdoc} */
	public function getModalRecordsListSourceFields(QueryGenerator $queryGenerator, Vtiger_Module_Model $baseModule, $popupFields)
	{
		foreach ($baseModule->getFieldsByType('email') as $item) {
			$popupFields[$item->getName()] = $item->getName();
		}
		return $popupFields;
	}
}
