<?php
/**
 * Report fields.
 *
 * @package   Modules
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Adrian Kon <a.kon@yetiforce.com>
 */
class ReportTemplate_ReportFields_Model extends ReportTemplate_BaseReportField_Model
{
	private string $sourceModuleName;
	private array $reportFields = [];
	private array $relatedItems;

	public function __construct(string $moduleName, array $relatedItems = [], ?Vtiger_Record_Model $reportRecordModel = null)
	{
		$this->relatedItems = $relatedItems;
		$this->sourceModuleName = $moduleName;
	}

	public function getFields(): array
	{
		$moduleFields = $this->getModuleFields($this->sourceModuleName);
		$moduleInventoryFields = $this->getInventoryFields($this->sourceModuleName);
		$reportFields = array_merge($moduleFields, $moduleInventoryFields);
		foreach ($this->relatedItems as $item) {
			if (isset($item['isRelatedAdvModule']) && $item['isRelatedAdvModule']) {
				$advRelatedModuleFields = $this->getAdvRelatedModuleFields($this->sourceModuleName, $item['relatedItem']);
				$reportFields = array_merge($reportFields, $advRelatedModuleFields);
			} elseif (App\Module::getModuleId($item['relatedItem'])) {
				$moduleReferenceFields = $this->getReferenceReportFields($item['sourceModule'], $item['relatedItem']);
				$reportFields = array_merge($reportFields, $moduleReferenceFields);
			} else {
				$relationFields = $this->getRelationFields($item['relatedItem']);
				$reportFields = array_merge($reportFields, $relationFields);
			}
		}
		return $reportFields;
	}

	public function getReferenceReportFields(string $sourceModule, string $referenceModule): array
	{
		$referenceModuleFields = [];
		$moduleModel = Vtiger_Module_Model::getInstance($sourceModule);
		$fieldTypes = 'Users' === $referenceModule ? 'owner' : \Vtiger_Field_Model::$referenceTypes;
		foreach ($moduleModel->getFieldsByType($fieldTypes) as $parentFieldName => $field) {
			if ('owner' === $field->getFieldDataType()) {
				$relatedModule = 'Users';
			} else {
				$referenceModuleList = $field->getReferenceList();
				$relatedModule = \in_array($referenceModule, $referenceModuleList) ? $referenceModule : null;
			}
			if ($referenceModule === $relatedModule && App\Privilege::isPermitted($relatedModule)) {
				$parentFieldNameLabel = App\Language::translate($field->getLabel(), $sourceModule);
				$relatedModuleLang = App\Language::translate($relatedModule, $relatedModule);
				foreach (\Vtiger_Module_Model::getInstance($relatedModule)->getBlocks() as $blockModel) {
					foreach ($blockModel->getFields() as $fieldName => $fieldModel) {
						if ($fieldModel->isViewable()) {
							$translatedModuleName = App\Language::translate($sourceModule, $sourceModule);
							$labelGroup = "[{$translatedModuleName}] {$parentFieldNameLabel}: ($relatedModuleLang) " . App\Language::translate($blockModel->get('label'), $relatedModule);
							$reportField = $this->createReportField($fieldModel);
							$reportField->setLabel(sprintf(
								'[%s] %s',
								$relatedModuleLang,
								App\Language::translate($fieldModel->getLabel(), $relatedModule),
							));
							$reportField->setExpressionField("{$fieldName}:{$relatedModule}:{$parentFieldName}");
							$referenceModuleFields[$labelGroup][] = $reportField;
						}
					}
				}
				$baseEntityField = $this->getBaseIdModuleField($relatedModule);
				$baseExpression = "{$baseEntityField->getFieldName()}:{$relatedModule}:{$parentFieldName}";
				$baseEntityField->setExpressionField($baseExpression);
				$baseEntityField->setLabel(sprintf(
					'[%s] %s',
					$relatedModuleLang,
					App\Language::translate('LBL_ID', 'ReportTemplate'),
				));
				$referenceModuleFields = $this->unshiftField($referenceModuleFields, $baseEntityField);
			}
		}
		return $referenceModuleFields;
	}

	private function getModuleFields(string $moduleName): array
	{
		$moduleFields = [];
		if (App\Privilege::isPermitted($moduleName)) {
			$moduleModel = Vtiger_Module_Model::getInstance($moduleName);
			foreach ($moduleModel->getFields() as $fieldModel) {
				if ($fieldModel->isActiveField() && 'reference' !== $fieldModel->getFieldDataType()) {
					$blockLabel = $this->getTranslatedBlockName($moduleName, $fieldModel->getBlockName());
					$translatedModuleName = $this->getTranslatedModuleName($moduleName);
					$reportField = $this->createReportField($fieldModel);
					$labelGroup = "[{$translatedModuleName}] {$blockLabel}";
					$moduleFields[$labelGroup][] = $reportField;
				}
			}
			$baseEntityField = $this->getBaseIdModuleField($moduleName);
			$baseEntityField->setExpressionField($baseEntityField->getFieldName());
			$moduleFields = $this->unshiftField($moduleFields, $baseEntityField);
		}
		return $moduleFields;
	}

	private function getInventoryFields(string $moduleName): array
	{
		$inventoryFields = [];
		$moduleModel = Vtiger_Module_Model::getInstance($moduleName);
		if ($moduleModel->isInventory() && $moduleModel->isActive()) {
			$translatedModuleName = App\Language::translate($moduleName, $moduleName);
			$inventoryModel = \Vtiger_Inventory_Model::getInstance($moduleName);
			foreach ($inventoryModel->getFields() as $fieldName => $fieldModel) {
				$fieldNameAndModule = "{$fieldName}:{$moduleName}:INVENTORY";
				if (!\in_array($fieldNameAndModule, $this->reportFields)) {
					$blockName = App\Language::translate('LBL_INVENTORY_BLOCK', 'ReportTemplate');
					$reportField = ReportTemplate_FieldReport_Model::fromInventory($fieldModel);
					$reportField->setLabel(sprintf(
						'[%s] %s',
						$translatedModuleName,
						App\Language::translate($fieldModel->getLabel(), $moduleName),
					));
					$reportField->setBlockName($blockName);
					$reportField->setExpressionField("{$fieldModel->getName()}:{$moduleName}:INVENTORY");

					$labelGroup = "$translatedModuleName $blockName";
					$inventoryFields[$labelGroup][] = $reportField;
				}
			}
		}
		return $inventoryFields;
	}

	private function getRelationFields(int $relationId): array
	{
		$relationFields = [];
		$relation = Vtiger_Relation_Model::getInstanceById($relationId);
		$relationModuleModel = $relation->getRelationModuleModel();
		$relatedModuleName = $relationModuleModel->getName();
		$translatedModuleName = App\Language::translate($relatedModuleName, $relatedModuleName);
		foreach ($relationModuleModel->getFields() as $fieldModel) {
			if ($fieldModel->isActiveField()) {
				$reportField = ReportTemplate_FieldReport_Model::from($fieldModel);
				$reportField->setExpressionField(sprintf(
					'%s:%s:%s:M2M',
					$fieldModel->getName(),
					$relatedModuleName,
					$relationId,
				));
				$reportField->setLabel(sprintf(
					'[%s] %s',
					$translatedModuleName,
					App\Language::translate($fieldModel->getLabel(), $relatedModuleName),
				));
				$blockLabel = App\Language::translate($fieldModel->getBlockName(), $relatedModuleName);
				$relationLabel = App\Language::translate($relation->get('label'), $relatedModuleName);
				$labelGroup = "[{$relationLabel}] {$blockLabel}";
				$relationFields[$labelGroup][] = $reportField;
			}
		}
		$baseEntityField = $this->getBaseIdModuleField($relatedModuleName);
		$baseEntityField->setExpressionField(sprintf(
			'%s:%s:%s:M2M',
			$baseEntityField->getFieldName(),
			$relatedModuleName,
			$relationId
		));

		return $this->unshiftField($relationFields, $baseEntityField);
	}

	private function getAdvRelatedModuleFields(string $inventoryModule, string $relatedInvModule)
	{
		$inventoryFields = [];
		$moduleModel = Vtiger_Module_Model::getInstance($inventoryModule);
		if ($moduleModel->isInventory() && $moduleModel->isActive()) {
			$inventoryModel = \Vtiger_Inventory_Model::getInstance($inventoryModule);
			if ($inventoryModel->isField('name')) {
				$inventoryFieldName = $inventoryModel->getField('name');
				foreach ($inventoryFieldName->getModules() as $inventoryModuleName) {
					if ($relatedInvModule === $inventoryModuleName && \App\Module::isModuleActive($inventoryModuleName)) {
						$advancedModuleModel = Vtiger_Module_Model::getInstance($inventoryModuleName);
						foreach ($advancedModuleModel->getFields() as $fieldName => $fieldModel) {
							$fieldModel = Vtiger_Field_Model::getInstance($fieldName, $advancedModuleModel);
							if ($fieldModel->isActiveField()) {
								$blockLabel = $this->getTranslatedBlockName($relatedInvModule, $fieldModel->getBlockName());
								$translatedModuleName = $this->getTranslatedModuleName($relatedInvModule);
								$reportField = $this->createReportField($fieldModel);
								$reportField->setExpressionField("{$fieldModel->getName()}:{$inventoryModuleName}:name:INVENTORY");
								$reportField->setLabel(sprintf(
									'[%s] %s',
									$translatedModuleName,
									App\Language::translate($fieldModel->getLabel(), $inventoryModuleName),
								));
								$labelGroup = "[{$translatedModuleName}] {$blockLabel}";
								$inventoryFields[$labelGroup][] = $reportField;
							}
						}
						$baseEntityField = $this->getBaseIdModuleField($inventoryModuleName);
						$baseEntityField->setExpressionField("{$baseEntityField->getFieldName()}:{$inventoryModuleName}:name:INVENTORY");
						$inventoryFields = $this->unshiftField($inventoryFields, $baseEntityField);
					}
				}
			}
		}
		return $inventoryFields;
	}
}
