<?php

/**
 * Render template with data.
 *
 * @package   ReportTemplates
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Paweł Karczmarcyk <p.karczmarczyk@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

use App\Controller\View\Page;
use App\Exceptions\NoPermittedToRecord;
use App\Privilege;
use App\Report\Builder\QueryBuilder;
use App\Report\Provider\FieldsProvider;
use App\Report\Provider\ReportProvider;
use App\Report\Report;
use App\Report\Sanitizer\FilterSanitizer;
use App\Request;
use App\User;

class ReportTemplate_Preview_View extends Page
{
	/** {@inheritdoc} */
	public function checkPermission(Request $request): void
	{
		$recordId = $request->isEmpty('record', true) ? 0 : $request->getInteger('record');
		if (!Privilege::isPermitted($request->getModule(), 'EditView', $recordId)) {
			throw new NoPermittedToRecord('ERR_NO_PERMISSIONS_FOR_THE_RECORD', 406);
		}
	}

	/**
	 * @param Request $request
	 *
	 * @return void
	 */
	public function process(Request $request): void
	{
		if ($request->has('query')) {
			[$previewData, $previewHeaders] = $this->getDataByQuery($request->getArray('query'));
		} else {
			[$previewData, $previewHeaders] = $this->getDataById($request->getInteger('record'));
		}

		$viewer = $this->getViewer($request);
		$viewer->assign('PREVIEW_DATA', $previewData);
		$viewer->assign('PREVIEW_HEADERS', $previewHeaders);
		echo $viewer->view('Preview.tpl', $request->getModule(), true);
	}

	private function getDataById(int $reportId)
	{
		$userId = User::getCurrentUserId();

		$dataProvider = Report::get(ReportProvider::class);
		$fieldsProvider = Report::get(FieldsProvider::class);

		$previewData = $dataProvider->provide($reportId, $userId, true);
		$previewHeaders = $fieldsProvider->provide($reportId, \count($previewData) > 0 ? $previewData[0] : []);

		return [$previewData, $previewHeaders];
	}

	private function getDataByQuery(array $data)
	{
		$moduleName = $data['base_module_id'];
		$expressions = $data['expressions'];
		$query = Report::get(QueryBuilder::class)->query($moduleName);
		$filters = Report::get(FilterSanitizer::class)->sanitize($query, $data['filter_group']);

		$previewData = Report::get(ReportProvider::class)->provideByData($moduleName, $expressions, $filters, User::getCurrentUserId(), true);

		$previewHeaders = Report::get(FieldsProvider::class)->provideByData(
			$moduleName,
			$expressions,
			\count($previewData) > 0 ? $previewData[0] : [],
		);

		return [$previewData, $previewHeaders];
	}
}
