<?php
/**
 * Transfer ownership modal action file.
 *
 * @package   Action
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

use App\Config;
use App\Controller\Action;
use App\Exceptions\NoPermitted;
use App\Language;
use App\Privilege;
use App\Purifier;
use App\Request;

/**
 * Transfer ownership modal action class.
 */
class Vtiger_TransferOwnership_Action extends Action
{
	/** {@inheritdoc} */
	public function checkPermission(Request $request)
	{
		$moduleName = $request->getModule();
		if ($request->isEmpty('record', true) || !$request->isEmpty('relatedModule', true)) {
			if ($request->isEmpty('relatedModule', true)) {
				$moduleName = $request->getModule();
			} else {
				$moduleName = $request->get('relatedModule', Purifier::ALNUM);
			}
			$record = Vtiger_Record_Model::getCleanInstance($moduleName);
			$permission = $record->isPermitted('EditView') && $record->isPermitted('MassTransferOwnership');
		} else {
			$record = Vtiger_Record_Model::getInstanceById($request->getInteger('record'), $moduleName);
			$permission = $record->isEditable() && $record->isPermitted('DetailTransferOwnership');
		}
		if (!$permission) {
			throw new NoPermitted('LBL_PERMISSION_DENIED', 406);
		}
	}

	/** {@inheritdoc} */
	public function process(Request $request)
	{
		if ($request->isEmpty('relatedModule', true)) {
			$moduleName = $request->getModule();
		} else {
			$moduleName = $request->get('relatedModule', Purifier::ALNUM);
		}
		$transferOwnerId = $request->getInteger('transferOwnerId');
		$relatedModules = $request->getByType('related_modules', 'Text');

		$modelClassName = Vtiger_Loader::getComponentClassName('Model', 'TransferOwnership', $moduleName);
		/** @var Vtiger_TransferOwnership_Model */
		$transferModel = new $modelClassName();

		if ($request->isEmpty('record', true) || !$request->isEmpty('relatedModule', true)) {
			if ($request->isEmpty('relatedModule', true)) {
				$recordIds = Vtiger_Mass_Action::getRecordsListFromRequest($request);
			} else {
				$recordIds = Vtiger_RelationAjax_Action::getRecordsListFromRequest($request);
			}
			$configMaxTransferRecords = Config::performance('maxMassTransferOwnershipRecords');
			if (\count($recordIds) > $configMaxTransferRecords) {
				$response = new Vtiger_Response();
				$response->setResult(['notify' => ['text' => Language::translateArgs('LBL_SELECT_UP_TO_RECORDS', '_Base', $configMaxTransferRecords), 'type' => 'error']]);
				$response->emit();
				return;
			}
		} else {
			$recordIds = [$request->getInteger('record')];
		}
		if (!empty($recordIds)) {
			$transferModel->transferRecordsOwnership($moduleName, $transferOwnerId, $recordIds);
		}
		if (!empty($relatedModules)) {
			foreach ($relatedModules as $relatedData) {
				$explodedData = explode('::', $relatedData);
				$relatedModule = current($explodedData);
				$relatedModuleRecordIds = $transferModel->getRelatedModuleRecordIds($moduleName, $recordIds, $explodedData);
				if (!empty($relatedModuleRecordIds)) {
					$transferModel->transferRecordsOwnership($relatedModule, $transferOwnerId, $relatedModuleRecordIds);
				}
			}
		}
		$redirect = '';
		if (!$request->isEmpty('record', true) && !Privilege::isPermitted($moduleName, 'DetailView', $request->getInteger('record'))) {
			$redirect = Vtiger_Module_Model::getInstance($moduleName)->getDefaultUrl();
		}
		$response = new Vtiger_Response();
		$response->setResult(['success' => true, 'redirect' => $redirect]);
		$response->emit();
	}
}
