<?php
/**
 * Export to csv - file.
 *
 * @package Model
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Adrian Kon <a.kon@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

use App\Config;
use App\Db\Query;
use App\Export\Records;

/**
 * Export to csv - class.
 */
class Vtiger_ExportToCsv_Model extends Records
{
	/** @var string File extension. */
	protected $fileExtension = 'csv';

	/**
	 * Function exports the data based on the mode.
	 *
	 * @return void
	 */
	public function exportData(): void
	{
		$tmpFile = $this->exportDataToTmpFile();

		$this->output($tmpFile);
	}

	/** {@inheritDoc} */
	public function exportDataToTmpFile(): string
	{
		$tmpPath = tempnam(ROOT_DIRECTORY . DIRECTORY_SEPARATOR . Config::main('tmp_dir'), 'xls');
		$this->registerTmpFile($tmpPath);

		$output = fopen($tmpPath, 'w');
		fputcsv($output, $this->getHeaders());
		foreach ($this->getEntries() as $row) {
			fputcsv($output, $row);
		}
		fclose($output);

		return $tmpPath;
	}

	/**
	 * Get entires.
	 *
	 * @return array
	 */
	public function getEntries(): array
	{
		$entries = [];
		$addInventoryData = $this->fullData && $this->moduleInstance->isInventory();
		if ($addInventoryData) {
			$inventoryModel = Vtiger_Inventory_Model::getInstance($this->moduleName);
			$inventoryFields = $inventoryModel->getFields();
			$inventoryTable = $inventoryModel->getDataTableName();
		}
		$rowsCounter = 0;
		$dataReader = $this->getExportQuery()->createCommand()->query();
		while ($row = $dataReader->read()) {
			$sanitizedRow = $this->sanitizeValues($row);
			if ($addInventoryData) {
				$sanitizedRow[] = $rowsCounter++;
				$rows = (new Query())->from($inventoryTable)->where(['crmid' => $row['id']])->orderBy('seq')->all();
				if ($rows) {
					foreach ($rows as &$row) {
						$sanitizedInventoryRow = $this->sanitizeInventoryValues($row, $inventoryFields);
						$entries[] = array_merge($sanitizedRow, $sanitizedInventoryRow);
					}
				} else {
					$entries[] = $sanitizedRow;
				}
			} else {
				$entries[] = $sanitizedRow;
			}
		}
		$dataReader->close();
		return $entries;
	}

	/**
	 * Function that outputs the exported file.
	 *
	 * @param string $tmpPath
	 */
	protected function output(string $tmpPath)
	{
		$file = fopen($tmpPath, 'r');
		fpassthru($file);
		fclose($file);
	}
}
