<?php
/**
 * Mass edit modal view file.
 *
 * @package   View
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */

use App\Condition;
use App\Controller\Modal;
use App\Exceptions\NoPermitted;
use App\Json;
use App\ModuleHierarchy;
use App\Request;

/**
 * Mass edit modal view class.
 */
class Vtiger_MassEditModal_View extends Modal
{
	/** {@inheritdoc} */
	public $modalSize = 'modal-lg';

	/** {@inheritdoc} */
	public $modalIcon = 'fa fa-edit';

	/** {@inheritdoc} */
	public $showFooter = false;

	/** {@inheritdoc} */
	protected $pageTitle = 'LBL_MASS_EDITING';

	/** {@inheritdoc} */
	public function checkPermission(Request $request)
	{
		$userPriviligesModel = Users_Privileges_Model::getCurrentUserPrivilegesModel();
		if (!$userPriviligesModel->hasModuleActionPermission($request->getModule(), 'MassEdit')) {
			throw new NoPermitted('LBL_PERMISSION_DENIED', 406);
		}
	}

	/**
	 * Users list modal.
	 *
	 * @param Request $request
	 *
	 * @return void
	 */
	public function process(Request $request): void
	{
		$moduleName = $request->getModule();
		$viewer = $this->getViewer($request);
		$moduleModel = Vtiger_Module_Model::getInstance($moduleName);
		$recordStructureInstance = Vtiger_RecordStructure_Model::getInstanceForModule($moduleModel, Vtiger_RecordStructure_Model::RECORD_STRUCTURE_MODE_MASSEDIT);
		$viewer->assign('MODULE_NAME', $moduleName);
		$viewer->assign('MODE', 'massedit');
		$viewer->assign('CVID', $request->getByType('viewname', 2));
		$viewer->assign('SELECTED_IDS', $request->getArray('selected_ids', 2));
		$viewer->assign('EXCLUDED_IDS', $request->getArray('excluded_ids', 2));
		$viewer->assign('RECORD_STRUCTURE_MODEL', $recordStructureInstance);
		$viewer->assign('MODULE_MODEL', $moduleModel);
		$viewer->assign('RECORD_STRUCTURE', $recordStructureInstance->getStructure());
		$viewer->assign('MAPPING_RELATED_FIELD', Json::encode(ModuleHierarchy::getRelationFieldByHierarchy($moduleName)));
		$viewer->assign('LIST_FILTER_FIELDS', Json::encode(ModuleHierarchy::getFieldsForListFilter($moduleName)));
		$viewer->assign('OPERATOR', $request->getByType('operator'));
		$viewer->assign(
			'ALPHABET_VALUE',
			Condition::validSearchValue(
				$request->getByType('search_value', 'Text'),
				$moduleName,
				$request->getByType('search_key', 'Alnum'),
				$request->getByType('operator')
			)
		);
		$viewer->assign('SEARCH_KEY', $request->getByType('search_key', 'Alnum'));
		$viewer->assign('SEARCH_PARAMS', Condition::validSearchParams($moduleName, $request->getArray('search_params'), false));
		$advancedConditions = $request->has('advancedConditions') ? $request->getArray('advancedConditions') : [];
		if ($advancedConditions) {
			Condition::validAdvancedConditions($advancedConditions);
		}
		$viewer->assign('ADVANCED_CONDITIONS', $advancedConditions);
		$viewer->view('Modals/MassEditModal.tpl', $moduleName);
	}
}
