<?php
/**
 * Webservice standard container - Get record list file.
 *
 * @package API
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

namespace Api\WebserviceStandard\BaseModule;

use App\FieldCoordinatorTransformer\QueryGeneratorFieldTransformer;
use OpenApi\Annotations as OA;

/**
 * Webservice standard container - Get record list class.
 */
class RecordsList extends \Api\Core\BaseAction
{
	/** {@inheritdoc}  */
	public $allowedMethod = ['GET'];
	/** {@inheritdoc}  */
	public $allowedHeaders = [
		'x-parent-id',
		'x-fields', 'x-only-column', 'x-fields-params', 'x-fields-info', 'x-response-params',
		'x-condition', 'x-cv-id', 'x-row-offset', 'x-row-limit', 'x-order-by', 'x-row-count', 'x-raw-data',
	];
	/** @var \App\QueryGenerator Query generator instance. */
	protected $queryGenerator;
	/** @var \Vtiger_Field_Model[] Fields models instance. */
	protected $fields = [];
	/** @var array Related fields. */
	protected $relatedFields = [];
	/** @var string[] Custom filter fields, based on search_params from path. */
	protected $filterFields = [];
	/** @var array Permissions. */
	protected $permissions = [];
	/** @var array Field params. */
	protected $fieldParams = [];

	/**
	 * Get record list method.
	 *
	 * @api
	 *
	 * @return array
	 *
	 * @OA\Get(
	 *		path="/webservice/WebserviceStandard/{moduleName}/RecordsList",
	 *		summary="List of records",
	 *		description="Gets a list of records",
	 *		tags={"BaseModule"},
	 *		security={{"basicAuth" : {}, "ApiKeyAuth" : {}, "token" : {}}},
	 *		@OA\Parameter(name="moduleName", in="path", @OA\Schema(type="string"), description="Module name", required=true, example="Contacts"),
	 *		@OA\Parameter(name="conditions", in="query", description="Conditions", required=false,
	 *			@OA\Schema(ref="#/components/schemas/Conditions-Mix-For-Query-Generator"),
	 *		),
	 *		@OA\Parameter(name="x-encrypted", in="header", @OA\Schema(ref="#/components/schemas/Header-Encrypted")),
	 *		@OA\Parameter(name="x-session-info", in="header", @OA\Schema(ref="#/components/schemas/Header-Session-Info"), description="Get session life information"),
	 *		@OA\Parameter(name="x-session-uptime", in="header", @OA\Schema(ref="#/components/schemas/Header-Session-Uptime"), description="Whether to update session life"),
	 *		@OA\Parameter(name="x-raw-data", in="header", @OA\Schema(ref="#/components/schemas/Header-Raw-Data"), description="Get additional raw data"),
	 *		@OA\Parameter(name="x-row-limit", in="header", @OA\Schema(type="integer"), description="Get rows limit, default: 100", required=false, example=50),
	 *		@OA\Parameter(name="x-row-offset", in="header", @OA\Schema(type="integer"), description="Offset, default: 0", required=false, example=0),
	 *		@OA\Parameter(name="x-row-count", in="header", @OA\Schema(type="integer", enum={0, 1}), description="Whether to get the number of records", required=false, example=1),
	 *		@OA\Parameter(name="x-fields", in="header", description="JSON array in the list of fields to be returned in response", required=false,
	 *			@OA\JsonContent(type="array", example={"field_name_1", "field_name_2"}, @OA\Items(type="string")),
	 *		),
	 *		@OA\Parameter(name="x-condition", in="header", description="Conditions [Json format]", required=false,
	 *			@OA\JsonContent(ref="#/components/schemas/Conditions-Mix-For-Query-Generator"),
	 *		),
	 *		@OA\Parameter(name="x-only-column", in="header", @OA\Schema(type="integer", enum={0, 1}), description="Return only column names", required=false, example=1),
	 *		@OA\Parameter(name="x-fields-info", in="header", @OA\Schema(type="integer", enum={0, 1}), description="Return additional field details", required=false, example=1),
	 *		@OA\Parameter(name="x-response-params", in="header", description="The header contains information about additional data to be returned in the response [Json array], depends on header `x-response-params`", required=false,
	 *			@OA\JsonContent(type="array", @OA\Items(type="string", enum={"privileges", "dbStructure", "queryOperators"})),
	 *		),
	 *		@OA\Parameter(name="x-parent-id", in="header", @OA\Schema(ref="#/components/schemas/Header-Parent-Id"), description="Parent record id"),
	 * 		@OA\Parameter(name="x-fields-params", in="header", description="JSON array - list of fields to be returned in the specified way", required=false,
	 *			@OA\JsonContent(ref="#/components/schemas/Fields-Settings"),
	 *		),
	 *		@OA\Parameter(name="x-cv-id", in="header", @OA\Schema(type="integer"), description="Custom view ID", required=false, example=5),
	 *		@OA\Parameter(name="x-order-by", in="header", description="Set the sorted results by columns [Json format]", required=false,
	 * 			@OA\JsonContent(type="object", title="Sort conditions", description="Multiple or one condition for a query generator",
	 * 				example={"field_name_1" : "ASC", "field_name_2" : "DESC"},
	 * 				@OA\AdditionalProperties(type="string", title="Sort Direction", enum={"ASC", "DESC"}),
	 * 			),
	 *		),
	 *		@OA\Response(response=200, description="List of entries",
	 *			@OA\JsonContent(ref="#/components/schemas/BaseModule_Get_RecordsList_Response"),
	 *			@OA\XmlContent(ref="#/components/schemas/BaseModule_Get_RecordsList_Response"),
	 *		),
	 *		@OA\Response(response=400, description="Incorrect json syntax: x-fields",
	 *			@OA\JsonContent(ref="#/components/schemas/Exception"),
	 *			@OA\XmlContent(ref="#/components/schemas/Exception"),
	 *		),
	 *		@OA\Response(response=401, description="No sent token, Invalid token, Token has expired",
	 *			@OA\JsonContent(ref="#/components/schemas/Exception"),
	 *			@OA\XmlContent(ref="#/components/schemas/Exception"),
	 *		),
	 *		@OA\Response(response=403, description="`No permissions for module` OR `No permissions for custom view: x-cv-id`",
	 *			@OA\JsonContent(ref="#/components/schemas/Exception"),
	 *			@OA\XmlContent(ref="#/components/schemas/Exception"),
	 *		),
	 *		@OA\Response(response=405, description="Invalid method",
	 *			@OA\JsonContent(ref="#/components/schemas/Exception"),
	 *			@OA\XmlContent(ref="#/components/schemas/Exception"),
	 *		),
	 *),
	 * @OA\Schema(
	 *		schema="BaseModule_Get_RecordsList_Response",
	 *		title="Base module - Response action record list",
	 *		description="Module action record list response body",
	 *		type="object",
	 *		required={"status", "result"},
	 *		@OA\Property(property="status", type="integer", enum={0, 1}, description="A numeric value of 0 or 1 that indicates whether the communication is valid. 1 - success , 0 - error"),
	 *		@OA\Property(property="result", type="object", title="List of records",
	 *			required={"headers", "records", "permissions", "numberOfRecords", "isMorePages"},
	 *			@OA\Property(property="headers", type="object", title="Fields names", example={"field_name_1" : "Field label 1", "field_name_2" : "Field label 2", "assigned_user_id" : "Assigned user", "createdtime" : "Created time"},
	 * 				@OA\AdditionalProperties(type="string", description="Field name"),
	 *			),
	 *			@OA\Property(property="records", type="object", title="Records display details",
	 *				@OA\AdditionalProperties(type="object", ref="#/components/schemas/Record_Display_Details"),
	 *			),
	 *			@OA\Property(property="permissions", type="object", title="Records action permissions",
	 *				@OA\AdditionalProperties(type="object", title="Record action permissions",
	 *					required={"isEditable", "moveToTrash"},
	 *					@OA\Property(property="isEditable", type="boolean", example=true),
	 *					@OA\Property(property="moveToTrash", type="boolean", example=true),
	 *				),
	 *			),
	 *			@OA\Property(property="fieldsDetails", type="object", title="Field details for returned columns", description="Depends on header `x-fields-info`",
	 *				@OA\AdditionalProperties(type="object", ref="#/components/schemas/Field_Info"),
	 *			),
	 *			@OA\Property(property="rawData", type="object", title="Records raw details, dependent on the header `x-raw-data`",
	 *				@OA\AdditionalProperties(type="object", ref="#/components/schemas/Record_Raw_Details"),
	 *			),
	 * 			@OA\Property(property="numberOfRecords", type="integer", description="Number of records on the page", example=20),
	 * 			@OA\Property(property="isMorePages", type="boolean", description="There are more pages", example=true),
	 * 			@OA\Property(property="numberOfAllRecords", type="integer", description="Number of all records, dependent on the header `x-row-count`", example=54),
	 * 		),
	 *	),
	 */
	public function get(): array
	{
		$this->createQuery();
		$response = $this->getResponse();
		if ((int) $this->controller->request->getHeader('x-only-column')) {
			return $response;
		}
		$this->fieldParams = \App\Json::decode($this->controller->request->getHeader('x-fields-params')) ?: [];
		$isRawData = $this->isRawData();
		$limit = $this->queryGenerator->getLimit();
		if ($limit) {
			$this->queryGenerator->setLimit($limit + 1);
		}
		$query = $this->queryGenerator->createQuery();
		$dataReader = $query->createCommand()->query();
		while ($row = $dataReader->read()) {
			$response['records'][$row['id']] = $this->getRecordFromRow($row);
			$response['permissions'][$row['id']] = $this->permissions;
			if ($isRawData) {
				$response['rawData'][$row['id']] = $this->getRawDataFromRow($row);
			}
		}
		$dataReader->close();
		$response['numberOfRecords'] = \count($response['records']);
		$isMorePages = false;
		if ($limit && $response['numberOfRecords'] > $limit) {
			$key = array_key_last($response['records']);
			unset($response['records'][$key], $response['rawData'][$key]);
			$isMorePages = true;
		}
		$response['isMorePages'] = $isMorePages;
		if ($this->controller->request->getHeader('x-row-count')) {
			$response['numberOfAllRecords'] = $query->count();
		}
		if ($this->filterFields) {
			$response['filterFields'] = array_keys($this->filterFields);
		}
		return $response;
	}

	/**
	 * Building a database query for a list of records.
	 *
	 * @throws \Api\Core\Exception
	 */
	public function createQuery(): void
	{
		$moduleName = $this->controller->request->getModule();
		$this->queryGenerator = new \App\QueryGenerator($moduleName);
		if ($cvId = $this->controller->request->getHeader('x-cv-id')) {
			$cv = \App\CustomView::getInstance($moduleName);
			if (!$cv->isPermittedCustomView($cvId)) {
				throw new \Api\Core\Exception('No permissions for custom view: x-cv-id', 403);
			}
			$this->queryGenerator->initForCustomViewById($cvId);
		} else {
			$this->queryGenerator->initForDefaultCustomView(false, true);
		}

		$limit = 100;
		if ($requestLimit = $this->controller->request->getHeader('x-row-limit')) {
			$limit = (int) $requestLimit;
		}
		$offset = 0;
		if ($requestOffset = $this->controller->request->getHeader('x-row-offset')) {
			$offset = (int) $requestOffset;
		}
		$this->queryGenerator->setLimit($limit);
		$this->queryGenerator->setOffset($offset);
		$this->createQueryFieldsConditions();
	}

	/**
	 * Building a database query field and conditions for a list of records.
	 *
	 * @see self::$allowedHeaders `x-fields` header, addicted to this sudden demand
	 * @see self::$allowedHeaders `x-order-by` header, addicted to this sudden demand
	 * @see self::$allowedHeaders `x-condition` header, addicted to this sudden demand
	 * @see self::$allowedHeaders `conditions` path, addicted to this sudden demand
	 *
	 * @return void
	 */
	protected function createQueryFieldsConditions(): void
	{
		\Api\WebserviceStandard\Fields::loadWebserviceFields($this->queryGenerator->getModuleModel(), $this);
		if ($requestFields = $this->controller->request->getHeader('x-fields')) {
			if (!\App\Json::isJson($requestFields)) {
				throw new \Api\Core\Exception('Incorrect json syntax: x-fields', 400);
			}
			$this->queryGenerator->clearFields();
			foreach (\App\Json::decode($requestFields) as $field) {
				if (\is_array($field)) {
					$this->queryGenerator->addRelatedField($field);
				} else {
					$this->queryGenerator->setField($field);
				}
			}
		}
		if ($orderBy = $this->controller->request->getHeader('x-order-by')) {
			$orderBy = \App\Json::decode($orderBy);
			if (!empty($orderBy) && \is_array($orderBy)) {
				foreach ($orderBy as $fieldName => $sortFlag) {
					$field = $this->queryGenerator->getModuleField($fieldName);
					if (($field && $field->isActiveField()) || 'id' === $fieldName) {
						$this->queryGenerator->setOrder($fieldName, $sortFlag);
					}
				}
			}
		}
		$this->fields = $this->queryGenerator->getListViewFields();
		foreach ($this->queryGenerator->getRelatedFields() as $fieldInfo) {
			$this->relatedFields[$fieldInfo['relatedModule']][$fieldInfo['sourceField']][] = $fieldInfo;
		}
		if ($conditions = $this->controller->request->get('conditions')) {
			$this->addConditionsQuery($conditions);
		} elseif ($conditions = $this->controller->request->getHeader('x-condition')) {
			$this->addConditionsQuery(\App\Json::decode($conditions));
		}
	}

	/**
	 * Add a conditions to query generator.
	 *
	 * @param array $conditions
	 *
	 * @return void
	 */
	protected function addConditionsQuery(array $conditions): void
	{
		if (isset($conditions['fieldName'])) {
			$this->addConditionQuery($conditions);
		} else {
			foreach ($conditions as $condition) {
				$this->addConditionQuery($condition);
			}
		}
	}

	/**
	 * Add a condition to query generator.
	 *
	 * @param array $condition
	 *
	 * @return void
	 */
	protected function addConditionQuery(array $condition): void
	{
		[$fieldName, $moduleName, $sourceFieldName] = array_pad(explode(':', $condition['fieldName']), 3, false);
		if (empty($sourceFieldName)) {
			$this->queryGenerator->addCondition(
				$condition['fieldName'],
				$condition['value'],
				$condition['operator'],
				$condition['group'] ?? true,
				true
			);
		} else {
			$this->queryGenerator->addRelatedCondition([
				'relatedModule' => $moduleName,
				'relatedField' => $fieldName,
				'sourceField' => $sourceFieldName,
				'value' => $condition['value'],
				'operator' => $condition['operator'],
				'conditionGroup' => $condition['group'] ?? true
			]);
		}
	}

	/**
	 * Check if you send raw data.
	 *
	 * @return bool
	 */
	protected function isRawData(): bool
	{
		return 1 === (int) ($this->controller->headers['x-raw-data'] ?? 0);
	}

	/**
	 * Get record from row.
	 *
	 * @param array $row
	 *
	 * @return array
	 */
	protected function getRecordFromRow(array $row): array
	{
		$record = ['recordLabel' => \App\Record::getLabel($row['id'])];
		if ($this->fields) {
			$moduleModel = reset($this->fields)->getModule();
			$recordModel = $moduleModel->getRecordFromArray($row);
			$this->permissions = [
				'isEditable' => $recordModel->isEditable(),
				'moveToTrash' => $recordModel->privilegeToMoveToTrash(),
			];
			foreach ($this->fields as $fieldName => $fieldModel) {
				if (isset($row[$fieldName])) {
					$record[$fieldName] = $fieldModel->getUITypeModel()
						->getApiDisplayValue(
							$row[$fieldName],
							$recordModel,
							$this->fieldParams[$fieldName] ?? []
						);
				}
			}
		}
		if ($this->relatedFields) {
			foreach ($this->relatedFields as $relatedModuleName => $sourceFields) {
				$relatedModuleModel = \Vtiger_Module_Model::getInstance($relatedModuleName);
				foreach ($sourceFields as $sourceField => $fields) {
					/** @see \App\QueryGenerator::createColumnAlias */
					$columnName = QueryGeneratorFieldTransformer::combine('id', $relatedModuleName, $sourceField);
					$recordData = [
						'id' => $row[$columnName] ?? 0,
					];
					foreach ($fields as $fieldInfo) {
						$recordData[$fieldInfo['relatedField']] = $row[$fieldInfo['columnAlias']];
					}
					$extRecordModel = $relatedModuleModel->getRecordFromArray($recordData);
					foreach ($fields as $fieldInfo) {
						if ($relatedFieldModel = $extRecordModel->getField($fieldInfo['relatedField'])) {
							$record[$fieldInfo['columnAlias']] = $relatedFieldModel->getUITypeModel()
								->getApiDisplayValue($row[$fieldInfo['columnAlias']], $extRecordModel);
						}
					}
				}
			}
		}
		return $record;
	}

	/**
	 * Get column names.
	 *
	 * @see self::$allowedHeaders `x-cv-id` header, addicted to this sudden demand
	 *
	 * @return array
	 */
	protected function getColumnNames(): array
	{
		$headers = [];
		$selectedColumns = [];
		if ($cvId = $this->controller->request->getHeader('x-cv-id')) {
			$customViewModel = \CustomView_Record_Model::getInstanceById($cvId);
			$selectedColumns = $customViewModel->getSelectedFields();
		}
		if ($this->fields) {
			foreach ($this->fields as $fieldName => $fieldModel) {
				if ($fieldModel && $fieldModel->isViewable()) {
					$moduleName = $fieldModel->getModuleName();
					/** @see \App\QueryGenerator::createColumnAlias */
					$key = QueryGeneratorFieldTransformer::combine($fieldName, $moduleName);
					$fieldLabel = empty($selectedColumns[$key]) ? $fieldModel->getLabel() : $selectedColumns[$key];
					$headers[$fieldName] = \App\Language::translate($fieldLabel, $moduleName);
				}
			}
		}
		if ($this->relatedFields) {
			foreach ($this->relatedFields as $relatedModuleName => $sourceFields) {
				$moduleModel = \Vtiger_Module_Model::getInstance($relatedModuleName);
				foreach ($sourceFields as $sourceField => $fields) {
					foreach ($fields as $fieldInfo) {
						$fieldModel = $moduleModel->getFieldByName($fieldInfo['relatedField']);
						if ($fieldModel && $fieldModel->isViewable()) {
							/** @see \App\QueryGenerator::createColumnAlias */
							$key = QueryGeneratorFieldTransformer::combine($fieldInfo['relatedField'], $relatedModuleName, $sourceField);
							$fieldLabel = empty($selectedColumns[$key]) ? $fieldModel->getLabel() : $selectedColumns[$key];
							$headers[$fieldInfo['columnAlias']] = \App\Language::translate($fieldLabel, $relatedModuleName);
						}
					}
				}
			}
		}
		return $headers;
	}

	/**
	 * Get base response.
	 *
	 * @return array
	 */
	protected function getResponse(): array
	{
		$response = [
			'headers' => $this->getColumnNames(),
			'records' => [],
			'permissions' => [],
		];
		if ($this->controller->request->getHeader('x-fields-info')) {
			$response['fieldsDetails'] = $this->getFieldsDetails();
		}
		return $response;
	}

	/**
	 * Get fields details.
	 *
	 * @see self::$allowedHeaders `x-fields-info` header, addicted to this sudden demand
	 *
	 * @return array
	 */
	protected function getFieldsDetails(): array
	{
		$rows = [];
		foreach ($this->fields as $fieldName => $fieldModel) {
			if ($fieldModel && $fieldModel->isViewable()) {
				$rows[$fieldName] = Fields::getFieldInfo($fieldModel, $this);
			}
		}
		foreach ($this->relatedFields as $relatedModuleName => $sourceFields) {
			$moduleModel = \Vtiger_Module_Model::getInstance($relatedModuleName);
			foreach ($sourceFields as $fields) {
				foreach ($fields as $fieldInfo) {
					$fieldModel = $moduleModel->getFieldByName($fieldInfo['relatedField']);
					if ($fieldModel && $fieldModel->isViewable()) {
						$fieldModel->set('source_field_name', $fieldInfo['sourceField']);
						$rows[$fieldInfo['columnAlias']] = Fields::getFieldInfo($fieldModel, $this);
					}
				}
			}
		}
		foreach ($this->filterFields as $columName => $columDetail) {
			if (!isset($rows[$columName])) {
				if (\is_string($columDetail)) {
					$fieldModel = $this->queryGenerator->getModuleField($columDetail);
					if ($fieldModel && $fieldModel->isViewable()) {
						$rows[$columDetail] = Fields::getFieldInfo($fieldModel, $this);
					}
				} else {
					$moduleModel = \Vtiger_Module_Model::getInstance($columDetail['relatedModule']);
					$fieldModel = $moduleModel->getFieldByName($fieldInfo['relatedField']);
					if ($fieldModel && $fieldModel->isViewable()) {
						$fieldModel->set('source_field_name', $fieldInfo['sourceField']);
						$rows[$columName] = Fields::getFieldInfo($fieldModel, $this);
					}
				}
			}
		}
		return $rows;
	}

	/**
	 * Get raw data from row.
	 *
	 * @param array $row
	 *
	 * @return array
	 */
	protected function getRawDataFromRow(array $row): array
	{
		foreach ($this->fields as $fieldName => $fieldModel) {
			if (\array_key_exists($fieldName, $row)) {
				$row[$fieldName] = $fieldModel->getUITypeModel()->getRawValue($row[$fieldName]);
			}
		}
		if ($this->relatedFields) {
			foreach ($this->relatedFields as $relatedModuleName => $sourceFields) {
				$moduleModel = \Vtiger_Module_Model::getInstance($relatedModuleName);
				foreach ($sourceFields as $fields) {
					foreach ($fields as $fieldInfo) {
						if (\array_key_exists($fieldInfo['columnAlias'], $row)) {
							$fieldModel = $moduleModel->getFieldByName($fieldInfo['relatedField']);
							$row[$fieldInfo['columnAlias']] = $fieldModel->getUITypeModel()->getRawValue($row[$fieldInfo['columnAlias']]);
						}
					}
				}
			}
		}
		return $row;
	}
}
