<?php

/**
 * Basic abstract file for alternative login provider.
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 * @author Antoni Kiszka <a.kiszka@yetiforce.com>
 */

namespace App\Authenticator;

/**
 * Basic abstract class for alternative login provider.
 */
abstract class AbstractProvider extends \App\Base
{
	/**
	 * User-friendly name for the provider.
	 *
	 * @var string
	 */
	protected $label;

	/**
	 * Icon for the provider as a CSS class.
	 *
	 * @var string
	 */
	protected $icon;

	/**
	 * Additional information for the provider.
	 *
	 * @var string|null
	 */
	protected $additionalInfo;

	/**
	 * Field names to edit in settings.
	 *
	 * @var string[]
	 */
	protected $editFieldNames = [];

	/** @var array Fields that should be encrypted. */
	protected $encryptedFields = [];

	/**
	 * Constructor.
	 */
	public function __construct()
	{
		$this->setData(\App\UserAuth::getConfig($this->getName()));
	}

	/**
	 * List of fields that should be encrypted.
	 *
	 * @return array
	 */
	public function getEncryptedFields(): array
	{
		return $this->encryptedFields;
	}

	/**
	 * Check if the provider is active.
	 *
	 * @return bool
	 */
	public function isActive(): bool
	{
		return 1 === $this->get('isactive');
	}

	/**
	 * Check whether the provider was created in the database.
	 */
	public function exists(): bool
	{
		return !empty($this->get('id'));
	}

	/**
	 * Get provider name.
	 * Provider name | File name.
	 * Max lenght: 50 characters.
	 *
	 * @return string
	 */
	public function getName(): string
	{
		return substr(strrchr(static::class, '\\'), 1);
	}

	/**
	 * Get a friendly name for the provider.
	 *
	 * @return string
	 */
	public function getLabel(): string
	{
		return $this->label;
	}

	/**
	 * Get an icon for the provider as a CSS class.
	 *
	 * @return string
	 */
	public function getIcon(): string
	{
		return $this->icon;
	}

	/**
	 * Get additional information for the provider.
	 * This information will be displayed in an alert at the top of the edit modal.
	 *
	 * @return string|null
	 */
	public function getAdditionalInfo(): ?string
	{
		return \App\Language::translate($this->additionalInfo, 'Settings:UserAuth');
	}

	/**
	 * Function to get Edit view url.
	 *
	 * @return string Url
	 */
	public function getEditViewUrl(): string
	{
		$model = \Settings_Vtiger_Module_Model::getInstance('Settings:UserAuth');
		return 'index.php?module=' . $model->getName() . '&parent=' . $model->getParentName() . "&view=Edit&provider={$this->getName()}";
	}

	/**
	 * Get the fields to edit in settings.
	 *
	 * @return array
	 */
	public function getEditFields(): array
	{
		$fields = [];
		foreach ($this->editFieldNames as $fieldName) {
			$fields[$fieldName] = $this->getFieldInstanceByName($fieldName);
		}

		return $fields;
	}

	/** {@inheritDoc} */
	public function get($fieldName)
	{
		$value = '';
		if ($this->has($fieldName)) {
			$value = parent::get($fieldName);
		} elseif ($this->get('config')) {
			$value = parent::get('config')[$fieldName] ?? '';
		}

		return $value;
	}

	/**
	 * Authentication process.
	 *
	 * @param string $password Reference added for anonymization as SensitiveParameter is not supported for PHP < 8.2
	 * @param string $userName
	 *
	 * @return bool
	 */
	public function verify(#[\SensitiveParameter] string &$password, string $userName): bool
	{
		return false;
	}

	/**
	 * Check if MFA verification is necessary.
	 *
	 * @return bool
	 */
	public function isMFA(): bool
	{
		return false;
	}

	/**
	 * Checks if the provided code is valid.
	 * Authentication for MFA.
	 *
	 * @param string $code
	 *
	 * @return bool
	 */
	public function verifyMFA(string $code): bool
	{
		return false;
	}

	/** Authentication pre-process. */
	public function preProcess()
	{
	}

	/**
	 * Log in user.
	 *
	 * @return void
	 */
	public function logIn()
	{
	}

	/**
	 * Authentication post-process.
	 *
	 * @return void
	 */
	public function postProcess()
	{
	}

	/**
	 * Fields to edit in settings.
	 *
	 * @param string $name
	 *
	 * @return \Vtiger_Field_Model|null
	 */
	abstract public function getFieldInstanceByName(string $name): ?\Vtiger_Field_Model;
}
