<?php

/**
 * Main file to integration with KSeF.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF;

use App\BatchMethod;
use App\Db;
use App\Db\Importer;
use App\Db\Importers\Base;
use App\Db\Query;
use App\EventHandler;
use App\Integrations\KSeF\Model\Mapper\CostInvoiceMapper;
use App\Integrations\KSeF\Model\Mapper\SalesInvoiceMapper;
use App\Integrations\KSeF\Service\Logger\KSeFLogger;
use App\Log;
use App\Module;
use vtlib\Block;
use vtlib\Cron;
use vtlib\Field;

/**
 * Main class to integration with KSeF.
 */
class KSeF
{
	/** @var string Settings table name */
	public const SETTINGS_TABLE_NAME = 'i_#__ksef_settings';

	/** @var string Send queue table name */
	public const QUEUE_TABLE_NAME = 'i_#__ksef_queue';

	/** @var string Download request table name */
	public const DOWNLOAD_REQUEST_TABLE_NAME = 'i_#__ksef_download_request';

	/** @var string System information */
	public const SYSTEM_INFO = 'YetiForce';

	/** @var string Path to ZIP files storage */
	public const ZIP_PATH = 'cache/ksef/';

	/** @var KSeFLogger Logger instance */
	private static KSeFLogger $logger;

	/**
	 * Get logger instance.
	 *
	 * @return KSeFLogger
	 */
	public static function getLogger(): KSeFLogger
	{
		return self::$logger ??= new KSeFLogger();
	}

	public function isActive(): bool
	{
		return Db::getInstance()->isTableExists(self::QUEUE_TABLE_NAME);
	}

	public function activate(): void
	{
		$db = Db::getInstance();
		$transaction = $db->beginTransaction();
		try {
			$this->addRates();
			EventHandler::registerHandler('InvoiceQueueAfterUpdate', 'FInvoice_KSeF_Handler', 'FInvoice', '', 5, true, Module::getModuleId('FInvoice'));
			EventHandler::registerHandler('DetailViewBefore', 'FInvoice_KSeF_Handler', 'FInvoice', '', 5, true, Module::getModuleId('FInvoice'));

			Cron::register('LBL_KSEF_CERTIFICATE_EXPIRATION_NOTIFICATION', 'App\Integrations\KSeF\Service\Cron\CertificateExpiration', 86400, null);
			Cron::register('LBL_KSEF_BATCH_SEND_NEW_INVOICES', 'App\Integrations\KSeF\Service\Cron\QueueBatchSend', 120, null);
			Cron::register('LBL_KSEF_BATCH_UPDATE_INVOICES_STATUS', 'App\Integrations\KSeF\Service\Cron\QueueBatchUpdate', 120, null);
			Cron::register('LBL_KSEF_IMPORT_INVOICES_DOWNLOAD', 'App\Integrations\KSeF\Service\Cron\ImportDownload', 600, null);
			Cron::register('LBL_KSEF_IMPORT_INVOICES_REQUEST', 'App\Integrations\KSeF\Service\Cron\ImportRequest', 1800, null);

			$this->addToProfiles();

			$module = 'FInvoiceCost';
			$targetModule = \vtlib\Module::getInstance($module);
			$sourceModule = \vtlib\Module::getInstance($module);
			$targetModule->setRelatedList($sourceModule, $sourceModule->name, ['ADD,SELECT'], 'getRelatedList');

			$this->createTable();
			$this->addMapping();
			$this->addFields();

			(new BatchMethod(['method' => '\App\UserPrivilegesFile::recalculateAll', 'params' => []]))->save();
			$transaction->commit();
		} catch (\Throwable $e) {
			Log::error($e->__toString(), 'KSeF');
			$transaction->rollBack();
			throw $e;
		}
	}

	private function addRates(): void
	{
		$baseRecordModel = new \Settings_Inventory_Record_Model();
		$baseRecordModel->setType('Taxes');
		$data = [
			['name' => 'VAT', 'value' => 23.00, 'status' => 0, 'default' => 1, 'show_name' => 0],
			['name' => '8%', 'value' => 8.00, 'status' => 0, 'default' => 0, 'show_name' => 0],
			['name' => '5%', 'value' => 5.00, 'status' => 0, 'default' => 0, 'show_name' => 0],
			['name' => '0%', 'value' => 0.00, 'status' => 0, 'default' => 0, 'show_name' => 0],
			['name' => 'NP', 'value' => 0.00, 'status' => 0, 'default' => 0, 'show_name' => 1],
			['name' => 'ZW', 'value' => 0.00, 'status' => 0, 'default' => 0, 'show_name' => 1],
		];
		$globalTaxes = \Vtiger_Inventory_Model::getGlobalTaxes();
		$taxes = array_column($globalTaxes, 'value', 'name');
		foreach ($data as $tax) {
			['name' => $name, 'value' => $taxValue] = $tax;
			if (($taxValue && \in_array($taxValue, $taxes)) || (empty($taxValue) && isset($taxes[$name]))) {
				continue;
			}
			$recordModel = clone $baseRecordModel;
			foreach ($tax as $key => $value) {
				$recordModel->set($key, $value);
			}
			$recordModel->save();
		}
	}

	private function addToProfiles(): void
	{
		$db = Db::getInstance();

		$actions = [
			['type' => 'add', 'name' => 'KSeF', 'tabsData' => [
				Module::getModuleId('FInvoice'),
			], 'permission' => 1]
		];
		foreach ($actions as $action) {
			$key = (new Query())->select(['actionid'])->from('vtiger_actionmapping')->where(['actionname' => $action['name']])->limit(1)->scalar();
			if (empty($key)) {
				$securitycheck = 0;
				$key = $db->getUniqueID('vtiger_actionmapping', 'actionid', false);
				$db->createCommand()->insert('vtiger_actionmapping', ['actionid' => $key, 'actionname' => $action['name'], 'securitycheck' => $securitycheck])->execute();
			}
			$permission = $action['permission'];
			$tabsData = $action['tabsData'];

			$dataReader = (new Query())->select(['profileid'])->from('vtiger_profile')->createCommand()->query();
			while (false !== ($profileId = $dataReader->readColumn(0))) {
				foreach ($tabsData as $tabId) {
					$isExists = (new Query())->from('vtiger_profile2utility')->where(['profileid' => $profileId, 'tabid' => $tabId, 'activityid' => $key])->exists();
					if (!$isExists) {
						$db->createCommand()->insert('vtiger_profile2utility', [
							'profileid' => $profileId, 'tabid' => $tabId, 'activityid' => $key, 'permission' => $permission,
						])->execute();
					}
				}
			}
			$dataReader->close();
		}
	}

	private function addFields(): void
	{
		$importerType = new Base();
		$fields = [
			[95, 3165, 'ksef_disable', 'u_yf_finvoice', 2, 56, 'ksef_disable', 'FL_KSEF_DISABLE', 0, 2, '', '-128,127', 0, 487, 1, 'C~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->tinyInteger(1), 'blockLabel' => 'LBL_KSEF', 'blockData' => ['label' => 'LBL_KSEF', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoice'],
			[95, 3166, 'ksef_status', 'u_yf_finvoice', 2, 16, 'ksef_status', 'FL_KSEF_STATUS', 0, 2, '', '80', 0, 487, 2, 'V~O', 1, 0, 'BAS', 2, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->stringType(80),
				'picklistValues' => ['LBL_STATUS_NEW', 'LBL_STATUS_IN_PROGRESS', 'LBL_STATUS_DONE', 'LBL_STATUS_REJECTED', 'LBL_STATUS_ERROR', 'LBL_STATUS_CANCELED'], 'blockLabel' => 'LBL_KSEF', 'blockData' => ['label' => 'LBL_KSEF', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoice'],
			[95, 3167, 'ksef_date', 'u_yf_finvoice', 2, 79, 'ksef_date', 'FL_KSEF_DATE', 0, 2, '', '', 0, 487, 2, 'DT~O', 1, 0, 'BAS', 2, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->dateTime(), 'blockLabel' => 'LBL_KSEF', 'blockData' => ['label' => 'LBL_KSEF', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoice'],
			[95, 3168, 'ksef_number', 'u_yf_finvoice', 2, 1, 'ksef_number', 'FL_KSEF_NUMBER', 0, 2, '', '80', 0, 487, 2, 'V~O', 1, 0, 'BAS', 2, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->stringType(80), 'blockLabel' => 'LBL_KSEF', 'blockData' => ['label' => 'LBL_KSEF', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoice'],
			[95, 3170, 'companyid', 'u_yf_finvoice', 2, 10, 'companyid', 'FL_MULTICOMPANY', 0, 2, '', '4294967295', 2, 488, 1, 'V~M', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->integer(10)->unsigned(),
				'relatedModules' => ['MultiCompany'], 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoice'],
			[95, 3171, 'split_payment', 'u_yf_finvoice', 2, 56, 'split_payment', 'FL_SPLIT_PAYMENT', 0, 2, '', '-128,127', 3, 488, 1, 'C~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->tinyInteger(1), 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoice'],
			[95, 3172, 'cash_method', 'u_yf_finvoice', 2, 56, 'cash_method', 'FL_CASH_METHOD', 0, 2, '', '-128,127', 5, 488, 1, 'C~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->tinyInteger(1), 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoice'],
			[95, 3173, 'reverse_charge', 'u_yf_finvoice', 2, 56, 'reverse_charge', 'FL_REVERSE_CHARGE', 0, 2, '', '-128,127', 4, 488, 1, 'C~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->tinyInteger(1), 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoice'],
			[95, 3174, 'margin_procedure', 'u_yf_finvoice', 2, 16, 'margin_procedure', 'FL_MARGIN_PROCEDURE', 0, 2, '', '255', 6, 488, 1, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->stringType(255),
				'picklistValues' => ['PLL_SECOND_HAND_GOODS', 'PLL_WORKS_OF_ART', 'PLL_TRAVEL_AGENCIES', 'PLL_COLLECTIBLES_AND_ANTIQUES'], 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoice'],
			[95, 3175, 'type_ksef', 'u_yf_finvoice', 2, 16, 'type_ksef', 'FL_TYPE_KSEF', 0, 2, '', '255', 1, 488, 1, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->stringType(255),
				'picklistValues' => ['PLL_VAT', /* 'PLL_ZAL', 'PLL_ROZ', */ 'PLL_UPR'], 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoice'],
			[95, 3176, 'tax_exemption_type', 'u_yf_finvoice', 2, 16, 'tax_exemption_type', 'FL_TAX_EXEMPTION_TYPE', 0, 2, '', '255', 7, 488, 1, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->stringType(255),
				'picklistValues' => ['PLL_ACT', 'PLL_DIRECTIVE', 'PLL_OTHER'], 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoice'],
			[95, 3177, 'tax_exemption_text', 'u_yf_finvoice', 2, 1, 'tax_exemption_text', 'FL_TAX_EXEMPTION_TEXT', 0, 2, '', '255', 8, 488, 1, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->stringType(255), 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoice'],
			[95, 3181, 'bankaccountid', 'u_yf_finvoice', 2, 10, 'bankaccountid', 'FL_BANK_ACCOUNT', 0, 2, '', '4294967295', 9, 488, 1, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->integer(10)->unsigned(),
				'relatedModules' => ['BankAccounts'], 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoice'],
			//			[95, 3183, 'settlement_invoice', 'u_yf_finvoice', 2, 10, 'settlement_invoice', 'FL_SETTLEMENT_INVOICE', 0, 2, '', '4294967295', 11, 488, 1, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->integer(10)->unsigned(),
			//				'relatedModules' => ['FInvoice'], 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoice'],

			[115, 3197, 'issue_time', 'u_yf_finvoicecost', 2, 5, 'issue_time', 'FL_ISSUE_TIME', 0, 2, '', '', 4, 388, 1, 'D~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->date(), 'blockLabel' => 'LBL_BASIC_DETAILS', 'blockData' => ['label' => 'LBL_BASIC_DETAILS', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 2, 'iscustom' => 0, 'icon' => ''], 'moduleName' => 'FInvoiceCost'],
			[115, 3198, 'ksef_date', 'u_yf_finvoicecost', 2, 79, 'ksef_date', 'FL_KSEF_DATE', 0, 2, '', '', 0, 490, 2, 'DT~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->dateTime(), 'blockLabel' => 'LBL_KSEF', 'blockData' => ['label' => 'LBL_KSEF', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoiceCost'],
			[115, 3199, 'ksef_number', 'u_yf_finvoicecost', 2, 1, 'ksef_number', 'FL_KSEF_NUMBER', 0, 2, '', '80', 0, 490, 2, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->stringType(80), 'blockLabel' => 'LBL_KSEF', 'blockData' => ['label' => 'LBL_KSEF', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoiceCost'],
			[115, 3200, 'full_address', 'u_yf_finvoicecost', 2, 1, 'full_address', 'FL_FULL_ADDRESS', 0, 2, '', '1024', 0, 490, 1, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->stringType(1024), 'blockLabel' => 'LBL_ADDRESS_INFORMATION', 'blockData' => ['label' => 'LBL_ADDRESS_INFORMATION', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoiceCost'],
			[115, 3202, 'invoice_cost_type_ksef', 'u_yf_finvoicecost', 2, 16, 'invoice_cost_type_ksef', 'FL_TYPE_KSEF', 0, 2, '', '255', 1, 491, 1, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->stringType(255),
				'picklistValues' => ['PLL_VAT', 'PLL_ZAL', 'PLL_ROZ', 'PLL_UPR', 'PLL_KOR', 'PLL_KOR_ZAL'], 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoiceCost'],
			[115, 3203, 'companyid', 'u_yf_finvoicecost', 2, 10, 'companyid', 'FL_MULTICOMPANY', 0, 2, '', '4294967295', 2, 491, 1, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->integer(10)->unsigned(),
				'relatedModules' => ['MultiCompany'], 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoiceCost'],
			[115, 3204, 'split_payment', 'u_yf_finvoicecost', 2, 56, 'split_payment', 'FL_SPLIT_PAYMENT', 0, 2, '', '-128,127', 3, 491, 1, 'C~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->tinyInteger(1), 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoiceCost'],
			[115, 3205, 'cash_method', 'u_yf_finvoicecost', 2, 56, 'cash_method', 'FL_CASH_METHOD', 0, 2, '', '-128,127', 4, 491, 1, 'C~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->tinyInteger(1), 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoiceCost'],
			[115, 3206, 'reverse_charge', 'u_yf_finvoicecost', 2, 56, 'reverse_charge', 'FL_REVERSE_CHARGE', 0, 2, '', '-128,127', 5, 491, 1, 'C~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->tinyInteger(1), 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoiceCost'],
			[115, 3207, 'margin_procedure', 'u_yf_finvoicecost', 2, 16, 'margin_procedure', 'FL_MARGIN_PROCEDURE', 0, 2, '', '255', 6, 491, 1, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->stringType(255), 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoiceCost'],
			[115, 3208, 'tax_exemption_type', 'u_yf_finvoicecost', 2, 16, 'tax_exemption_type', 'FL_TAX_EXEMPTION_TYPE', 0, 2, '', '255', 7, 491, 1, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->stringType(255), 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoiceCost'],
			[115, 3209, 'tax_exemption_text', 'u_yf_finvoicecost', 2, 1, 'tax_exemption_text', 'FL_TAX_EXEMPTION_TEXT', 0, 2, '', '255', 8, 491, 1, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->stringType(255), 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoiceCost'],
			[115, 3215, 'vat_id_a', 'u_yf_finvoicecost', 2, 1, 'vat_id_a', 'FL_VAT_ID', 0, 2, '', '50', 1, 492, 1, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->stringType(50), 'blockLabel' => 'LBL_ADDRESS_INFORMATION', 'blockData' => ['label' => 'LBL_ADDRESS_INFORMATION', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoiceCost'],
			[115, 3216, 'company_name_a', 'u_yf_finvoicecost', 2, 1, 'company_name_a', 'FL_COMPANY_NAME', 0, 2, '', '255', 2, 492, 1, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->stringType(255), 'blockLabel' => 'LBL_ADDRESS_INFORMATION', 'blockData' => ['label' => 'LBL_ADDRESS_INFORMATION', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoiceCost'],
			[115, 3227, 'bank_account', 'u_yf_finvoicecost', 2, 1, 'bank_account', 'FL_BANK_ACCOUNT', 0, 2, '', '255', 14, 491, 1, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->stringType(255),
				'relatedModules' => [], 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoiceCost'],
			[115, 3228, 'payment_url', 'u_yf_finvoicecost', 2, 17, 'payment_url', 'FL_PAYMENT_URL', 0, 2, '', '255', 13, 491, 1, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->stringType(512), 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoiceCost'],
			[115, 3229, 'correction_type', 'u_yf_finvoicecost', 2, 16, 'correction_type', 'FL_CORRECTION_TYPE', 0, 2, '', '255', 12, 491, 1, 'V~O', 1, 0, 'BAS', 1, '', 0, '', 0, 0, 0, 0, '', '', 'type' => $importerType->stringType(255),
				'picklistValues' => ['PLL_ORIGINAL', 'PLL_CORRECTIVE', 'PLL_OTHER'], 'blockLabel' => 'LBL_ADDITIONAL_INFO', 'blockData' => ['label' => 'LBL_ADDITIONAL_INFO', 'showtitle' => 0, 'visible' => 0, 'increateview' => 0, 'ineditview' => 0, 'indetailview' => 0, 'display_status' => 1, 'iscustom' => 1, 'icon' => ''], 'moduleName' => 'FInvoiceCost'],
		];

		foreach ($fields as $field) {
			$moduleName = $field['moduleName'];
			$moduleId = Module::getModuleId($moduleName);
			if (!$moduleId) {
				continue;
			}
			$isExists = (new Query())->from('vtiger_field')->where(['tablename' => $field[3], 'columnname' => $field[2], 'tabid' => $moduleId])->exists();
			if ($isExists) {
				continue;
			}

			$blockInstance = false;
			$blockId = (new Query())->select(['blockid'])->from('vtiger_blocks')->where(['blocklabel' => ($field['blockData']['label'] ?? $field['blockLabel']), 'tabid' => $moduleId])->scalar();
			if ($blockId) {
				$blockInstance = Block::getInstance($blockId, $moduleId);
			} elseif (!empty($field['blockData']) && \is_array($field['blockData'])) {
				$blockInstance = new Block();
				foreach ($field['blockData'] as $key => $value) {
					$blockInstance->{$key} = $value;
				}
				\Vtiger_Module_Model::getInstance($moduleName)->addBlock($blockInstance);
				$blockId = $blockInstance->id;
				$blockInstance = Block::getInstance($blockId, $moduleId);
			}
			if (!$blockInstance
				&& !($blockInstance = current(\Vtiger_Module_Model::getInstance($moduleName)->getBlocks()))) {
				continue;
			}
			$fieldInstance = new Field();
			$fieldInstance->column = $field[2];
			$fieldInstance->name = $field[6];
			$fieldInstance->label = $field[7];
			$fieldInstance->table = $field[3];
			$fieldInstance->uitype = $field[5];
			$fieldInstance->typeofdata = $field[15];
			$fieldInstance->readonly = $field[8];
			$fieldInstance->displaytype = $field[14];
			$fieldInstance->masseditable = $field[19];
			$fieldInstance->quickcreate = $field[16];
			$fieldInstance->columntype = $field['type'];
			$fieldInstance->presence = $field[9];
			$fieldInstance->maximumlength = $field[11];
			$fieldInstance->quicksequence = $field[17];
			$fieldInstance->info_type = $field[18];
			$fieldInstance->helpinfo = $field[20];
			$fieldInstance->summaryfield = $field[21];
			$fieldInstance->generatedtype = $field[4];
			$fieldInstance->defaultvalue = $field[10];
			$fieldInstance->fieldparams = $field[22];
			$blockInstance->addField($fieldInstance);
			if (!empty($field['picklistValues']) && (15 == $field[5] || 16 == $field[5] || 33 == $field[5])) {
				$fieldInstance->setPicklistValues($field['picklistValues']);
			}
			if (!empty($field['relatedModules']) && 10 == $field[5]) {
				$moduleList = (array) $field['relatedModules'];
				$fieldInstance->setRelatedModules($moduleList);
				foreach ($moduleList as $module) {
					$label = 'settlement_invoice' === $fieldInstance->name ? 'LBL_SETTLEMENT' : $blockInstance->module->name;
					$targetModule = \vtlib\Module::getInstance($module);
					$targetModule->setRelatedList($blockInstance->module, $label, ['Add'], 'getDependentsList', $fieldInstance->name);
				}
			}
		}
	}

	private function addMapping(): void
	{
		$this->addFInvoiceMapping();
		$this->addFInvoiceCostMapping();
	}

	private function createTable(): void
	{
		$importer = new Base();
		$db = Db::getInstance();

		$tables = [
			self::DOWNLOAD_REQUEST_TABLE_NAME => [
				'columns' => [
					'id' => $importer->primaryKeyUnsigned(),
					'status' => $importer->smallInteger(5)->notNull()->defaultValue(0),
					'settings_id' => $importer->integer()->notNull()->defaultValue(0),
					'reference_number' => $importer->stringType(),
					'date_input' => $importer->dateTime(),
					'date_mod' => $importer->dateTime()->notNull(),
					'module' => $importer->stringType(100),
					'key' => $importer->text(),
					'iv' => $importer->text(),
				],
				'columns_mysql' => [
					'status' => $importer->tinyInteger(5)->notNull()->defaultValue(0),
				],
				'engine' => 'InnoDB',
				'charset' => 'utf8mb4'
			],
			CostInvoiceMapper::TABLENAME => [
				'columns' => [
					'key' => $importer->stringType(100)->notNull(),
					'ksef_field' => $importer->stringType(80),
					'name' => $importer->stringType()->notNull(),
					'field' => $importer->stringType(500),
					'inventory_field' => $importer->stringType(80),
				],
				'index' => [
					['ksef_field_map_costinvoice_key_udx', 'key', true],
				],
				'engine' => 'InnoDB',
				'charset' => 'utf8mb4'
			],
			SalesInvoiceMapper::TABLENAME => [
				'columns' => [
					'key' => $importer->stringType(100)->notNull(),
					'ksef_field' => $importer->stringType(80),
					'name' => $importer->stringType()->notNull(),
					'field' => $importer->stringType(500),
					'inventory_field' => $importer->stringType(80),
				],
				'index' => [
					['ksef_field_map_salesinvoice_key_udx', 'key', true],
				],
				'engine' => 'InnoDB',
				'charset' => 'utf8mb4'
			],
			self::QUEUE_TABLE_NAME => [
				'columns' => [
					'id' => $importer->primaryKeyUnsigned(),
					'invoice_record_id' => $importer->integer(10)->unsigned()->notNull(),
					'settings_id' => $importer->integer(10)->unsigned()->notNull(),
					'status' => $importer->integer(2)->notNull(),
					'session_key' => $importer->stringType(60),
					'invoice_key' => $importer->stringType(60),
					'ksef_number' => $importer->stringType(60),
					'ordinal_number' => $importer->integer(10)->unsigned(),
					'mode' => $importer->integer(2),
					'date_input' => $importer->dateTime()->notNull(),
					'date_mod' => $importer->dateTime(),
					'date_acquisition' => $importer->dateTime(),
				],
				'engine' => 'InnoDB',
				'charset' => 'utf8mb4'
			],
		];
		$base = new Importer();
		$base->dieOnError = true;
		foreach ($tables as $tableName => $data) {
			if (!$db->isTableExists($tableName)) {
				$importer->tables = [$tableName => $data];
				$base->addTables($importer);
			}
		}
	}

	private function addFInvoiceMapping(): void
	{
		$mappingRows = [
			['//Fa/Adnotacje/P_16', 'P_16', 'Adnotacja „metoda kasowa"', 'cash_method', null],
			['//Fa/Adnotacje/P_17', 'P_17', 'Adnotacja „samofakturowanie"', null, null],
			['//Fa/Adnotacje/P_18', 'P_18', 'Adnotacja „odwrotne obciążenie"', 'reverse_charge', null],
			['//Fa/Adnotacje/P_18A', 'P_18A', 'Adnotacja „mechanizm podzielonej płatności"', 'split_payment', null],
			['//Fa/Adnotacje/PMarzy', 'PMarzy', 'Procedura marży', 'margin_procedure', null],
			['//Fa/Adnotacje/Zwolnienie/P_19', 'P_19', 'Znacznik zwolnienia podatku', 'tax_exemption_type', null],
			['//Fa/Adnotacje/Zwolnienie/P_19A', 'P_19A', 'Powód zwolnienia podatku (ustawa)', 'tax_exemption_text', null],
			['//Fa/Adnotacje/Zwolnienie/P_19B', 'P_19B', 'Powód zwolnienia podatku B (dyrektywa)', 'tax_exemption_text', null],
			['//Fa/Adnotacje/Zwolnienie/P_19C', 'P_19C', 'Powód zwolnienia podatku (inne)', 'tax_exemption_text', null],
			['//Fa/DodatkowyOpis', null, 'Datkowy opis - generowany automatycznie', 'description', null],
			['//Fa/DodatkowyOpis/Klucz', 'klucz', 'Dodatkowy opis 1: klucz', null, null],
			['//Fa/DodatkowyOpis/Wartosc', 'wartosc', 'Dodatkowy opis 1: wartość', null, null],
			['//Fa/FaWiersz/KursWaluty', 'KursWaluty', 'Pozycja: Kurs waluty', 'getCurrencyRate', 'currencyparam'],
			['//Fa/FaWiersz/P_11', 'p_11', 'Pozycja: wartość całkowita netto', null, 'net'],
			['//Fa/FaWiersz/P_11A', 'p_11A', 'Pozycja: wartość całkowita brutto', null, 'gross'],
			['//Fa/FaWiersz/P_11Vat', 'p_11Vat', 'Pozycja: wartość całkowita vat', null, 'tax'],
			['//Fa/FaWiersz/P_12', 'p_12', 'Pozycja: stawka podatku', 'getTaxPercent', 'tax_percent'],
			['//Fa/FaWiersz/P_7', 'p_7', 'Pozycja: nazwa', null, 'name'],
			['//Fa/FaWiersz/P_8A', 'p_8A', 'Pozycja: jednostka', null, 'unit'],
			['//Fa/FaWiersz/P_8B', 'p_8B', 'Pozycja: ilość/liczba', null, 'qty'],
			['//Fa/FaWiersz/P_9A', 'p_9A', 'Pozycja: wartość jednostkowa netto', null, 'price'],
			['//Fa/FaWiersz/P_10', 'P_10', 'Pozycja: Rabat', null, 'discount'],
			['//Fa/KodWaluty', 'kodWaluty', 'Waluta faktury', null, 'currency'],
			['//Fa/KursWalutyZ', 'KursWalutyZ', 'Kurs waluty dla zaliczki w obcej walucie', 'getCurrencyRate', 'currencyparam'],
			['//Fa/P_1', 'p_1', 'Data wystawienia', 'issue_time', null],
			['//Fa/P_15', 'p_15', 'Wartość całkowita', 'sum_gross', null],
			['//Fa/P_15Z', 'p_15Z', 'Wartość płatności', 'sum_gross', null],
			['//Fa/P_1M', 'p_1M', 'Miejsce wystawienia', null, null],
			['//Fa/P_2', 'p_2', 'Numer faktury', 'subject', null],
			['//Fa/P_6', 'p_6', 'Data sprzedaży', 'saledate', null],
			['//Fa/P_6_Do', 'p_6_Do', 'Data sprzedaży do', null, null],
			['//Fa/P_6_Od', 'p_6_Od', 'Data sprzedaży od', null, null],
			['//Fa/Platnosc/DataZaplaty', 'p_6Z', 'Data płatności', 'paymentdate', null],
			['//Fa/Platnosc/DataZaplatyCzesciowej', 'dataZaplatyCzesciowej', 'Data zapłaty częsciowej', 'paymentdate', null],
			['//Fa/Platnosc/FormaPlatnosci', 'FormaPlatnosci', 'Forma płatności', 'payment_methods', null],
			['//Fa/Platnosc/KwotaZaplatyCzesciowej', 'kwotaZaplatyCzesciowej', 'Kwota zapłaty częsciowej', 'sum_gross', null],
			['//Fa/Platnosc/TerminPlatnosci/Termin', 'Termin', 'Termin płatności', 'paymentdate', null],
			['//Fa/RachunekBankowy', 'RachunekBankowy', 'Powiązany rachunek bankowy', 'bankaccountid', null],
			['//Fa/RachunekBankowy/NazwaBanku', 'nazwaBanku', 'Nazwa banku', 'bank_name:BankAccounts:bankaccountid', null],
			['//Fa/RachunekBankowy/NrRB', 'nrRB', 'Numer rachunku bankowego', 'account_number:BankAccounts:bankaccountid', null],
			['//Fa/RachunekBankowy/SWIFT', 'swift', 'SWIFT', 'swift:BankAccounts:bankaccountid', null],
			['//Fa/RodzajFaktury', 'rodzajFaktury', 'Rodzaj faktury', 'type_ksef', null],
			['//Fa/Zamowienie/WartoscZamowienia', 'wartoscZamowienia', 'Suma zamówienia', null, 'sum_gross'],
			['//Fa/Zamowienie/ZamowienieWiersz/P_11NettoZ', 'p_11NettoZ', 'Pozycja: wartość całkowita netto', null, 'net'],
			['//Fa/Zamowienie/ZamowienieWiersz/P_11VatZ', 'p_11VatZ', 'Pozycja: wartość całkowita vat', null, 'tax'],
			['//Fa/Zamowienie/ZamowienieWiersz/P_12Z', 'p_12Z', 'Pozycja: stawka podatku', null, 'tax_percent'],
			['//Fa/Zamowienie/ZamowienieWiersz/P_7Z', 'p_7Z', 'Pozycja: nazwa', null, 'name'],
			['//Fa/Zamowienie/ZamowienieWiersz/P_8AZ', 'p_8AZ', 'Pozycja: jednostka', null, 'unit'],
			['//Fa/Zamowienie/ZamowienieWiersz/P_8BZ', 'p_8BZ', 'Pozycja: ilość/liczba', null, 'qty'],
			['//Fa/Zamowienie/ZamowienieWiersz/P_9AZ', 'p_9AZ', 'Pozycja: wartość jednostkowa netto', null, 'price'],
			['//Podmiot1/Adres/AdresL1', 'adresL1', 'Adres (linia 1) sprzedającego', '{"street":"addresslevel8a:MultiCompany:companyid","buildingNumber":"buildingnumbera:MultiCompany:companyid","localNumber":"localnumbera:MultiCompany:companyid","city":"addresslevel5a:MultiCompany:companyid","postCode":"addresslevel7a:MultiCompany:companyid"}', null],
			['//Podmiot1/Adres/KodKraju', 'kodKraju', 'Kod kraju sprzedającego', 'addresslevel1a:MultiCompany:companyid', null],
			['//Podmiot1/DaneIdentyfikacyjne/Nazwa', 'nazwa', 'Nazwa sprzedającego', 'company_name:MultiCompany:companyid', null],
			['//Podmiot1/DaneIdentyfikacyjne/NIP', 'nip', 'NIP sprzedającego', 'vat:MultiCompany:companyid', null],
			['//Podmiot1/DaneKontaktowe/Email', 'email', 'Osoba kontaktowa (sprzedający): email', null, null],
			['//Podmiot1/DaneKontaktowe/Telefon', 'telefon', 'Osoba kontaktowa (sprzedający): telefon', null, null],
			['//Podmiot2/Adres/AdresL1', 'adresL1', 'Adres (linia 1) kupującego', '{"street":"addresslevel8a","buildingNumber":"buildingnumbera","localNumber":"localnumbera","city":"addresslevel5a","postCode":"addresslevel7a"}', null],
			['//Podmiot2/Adres/KodKraju', 'kodKraju', 'Kod kraju kupującego', 'addresslevel1a', null],
			['//Podmiot2/DaneIdentyfikacyjne/Nazwa', 'nazwa', 'Nazwa kupującego', 'company_name_a', null],
			['//Podmiot2/DaneIdentyfikacyjne/NIP', 'nip', 'NIP kupującego', 'vat_id_a', null],
			['//Podmiot2/DaneKontaktowe/Email', 'email', 'Osoba kontaktowa (kupujący): email', null, null],
			['//Podmiot2/DaneKontaktowe/Telefon', 'telefon', 'Osoba kontaktowa (kupujący): telefon', null, null],
			['//Podmiot2/NrKlienta', 'nrKlienta', 'Numer klienta kupującego', null, null],
			['//Stopka/Informacje/StopkaFaktury', 'stopkaFaktury', 'Stopka: treść', 'description', null],
			['//Stopka/Rejestry/BDO', 'bdo', 'Stopka: BDO', null, null],
			['//Stopka/Rejestry/KRS', 'krs', 'Stopka: KRS', null, null],
			['//Stopka/Rejestry/PelnaNazwa', 'stopka_pelna_nazwa', 'Stopka: pełna nazwa firmy', null, null],
			['//Stopka/Rejestry/REGON', 'regon', 'Stopka: REGON', null, null],
			['fa_kursWalutyZW', 'kursWalutyZW', 'Kurs waluty płatności', 'getCurrencyRate', 'currencyparam'],
			['yf_company', null, 'Organizacja', 'companyid', null],
			['yf_finvoice_type', null, 'Rodzaj faktury (krajowa/UE/eksport)', null, null],
			['yf_ksef_number', null, 'Numer KSeF', 'ksef_number', null],
			['yf_payment_status', null, 'Status płatności', 'payment_status', null],
			['yf_payment_sum', null, 'Suma płatności', 'payment_sum', null],
//			['yf_settlement_invoice', null, 'Faktura rozliczająca', 'settlement_invoice', null],
			['yf_taxparam', null, 'Parametry podatku', null, 'taxparam'],
			['yf_total', null, 'Pozycja: wartość całkowita netto z rabatami', null, 'total']
		];

		$isExists = (new Query())->from(SalesInvoiceMapper::TABLENAME)->exists();
		if (!$isExists) {
			$dbCommand = Db::getInstance()->createCommand();
			$dbCommand->batchInsert(
				SalesInvoiceMapper::TABLENAME,
				['key', 'ksef_field', 'name', 'field', 'inventory_field'],
				$mappingRows
			)->execute();
		}
	}

	private function addFInvoiceCostMapping(): void
	{
		$mappingRows = [
			['yf_total', null, 'Pozycja: wartość całkowita netto z rabatami', null, 'total'],
			['//Fa/P_1M', 'p_1M', 'Miejsce wystawienia', null, null],
			['//Fa/P_6_Do', 'p_6_Do', 'Data sprzedaży do', null, null],
			['//Fa/P_6_Od', 'p_6_Od', 'Data sprzedaży od', null, null],
			['yf_taxparam', null, 'Parametry podatku', null, 'taxparam'],
			['//Fa/KodWaluty', 'kodWaluty', 'Waluta faktury', null, 'currency'],
			['yf_payment_sum', null, 'Suma płatności', null, null],
			['yf_finvoice_type', null, 'Rodzaj faktury (krajowa/UE/eksport)', null, null],
			['//Fa/FaWiersz/P_7', 'p_7', 'Pozycja: nazwa', null, 'name'],
			['yf_payment_status', null, 'Status płatności', null, null],
			['//Fa/FaWiersz/P_11', 'p_11', 'Pozycja: wartość całkowita netto', null, 'net'],
			['//Fa/FaWiersz/P_12', 'p_12', 'Pozycja: stawka podatku', null, 'tax_percent'],
			['//Fa/FaWiersz/P_8A', 'p_8A', 'Pozycja: jednostka', null, 'unit'],
			['//Fa/FaWiersz/P_8B', 'p_8B', 'Pozycja: ilość/liczba', null, 'qty'],
			['//Fa/FaWiersz/P_9A', 'p_9A', 'Pozycja: wartość jednostkowa netto', null, 'price'],
			['//Fa/Adnotacje/P_17', 'P_17', 'Adnotacja „samofakturowanie"', null, null],
			['//Fa/FaWiersz/P_11A', 'p_11A', 'Pozycja: wartość całkowita brutto', null, 'gross'],
			['//Fa/FaWiersz/P_8BZ', 'p_8BZ', 'Pozycja: ilość/liczba', null, 'qty'],
			['//Podmiot2/NrKlienta', 'nrKlienta', 'Numer klienta kupującego', null, null],
			['//Fa/FaWiersz/P_11Vat', 'p_11Vat', 'Pozycja: wartość całkowita vat', null, 'vat'],
			['//Stopka/Rejestry/BDO', 'bdo', 'Stopka: BDO', null, null],
			['//Fa/DodatkowyOpis/Klucz', 'klucz', 'Dodatkowy opis 1: klucz', null, null],
			['//Podmiot1/Adres/AdresL2', 'adresL2', 'Adres (linia 2) sprzedającego', null, null],
			['//Podmiot2/Adres/AdresL1', 'adresL1', 'Adres (linia 1) kupującego', null, null],
			['//Podmiot2/Adres/AdresL2', 'adresL2', 'Adres (linia 2) kupującego', null, null],
			['//Fa/DodatkowyOpis/Wartosc', 'wartosc', 'Dodatkowy opis 1: wartość', null, null],
			['//Podmiot1/DaneKontaktowe/Email', 'email', 'Osoba kontaktowa (sprzedający): email', null, null],
			['//Podmiot2/DaneKontaktowe/Email', 'email', 'Osoba kontaktowa (kupujący): email', null, null],
			['//Fa/Zamowienie/WartoscZamowienia', 'wartoscZamowienia', 'Suma zamówienia', null, 'sum_gross'],
			['//Podmiot1/DaneKontaktowe/Telefon', 'telefon', 'Osoba kontaktowa (sprzedający): telefon', null, null],
			['//Podmiot2/DaneKontaktowe/Telefon', 'telefon', 'Osoba kontaktowa (kupujący): telefon', null, null],
			['//Fa/Zamowienie/ZamowienieWiersz/P_7Z', 'p_7Z', 'Pozycja: nazwa', null, 'name'],
			['//Fa/Zamowienie/ZamowienieWiersz/P_12Z', 'p_12Z', 'Pozycja: stawka podatku', null, 'tax_percent'],
			['//Fa/Zamowienie/ZamowienieWiersz/P_8AZ', 'p_8AZ', 'Pozycja: jednostka', null, 'unit'],
			['//Fa/Zamowienie/ZamowienieWiersz/P_9AZ', 'p_9AZ', 'Pozycja: wartość jednostkowa netto', null, 'price'],
			['//Fa/Zamowienie/ZamowienieWiersz/P_11VatZ', 'p_11VatZ', 'Pozycja: wartość całkowita vat', null, 'vat'],
			['//Fa/Zamowienie/ZamowienieWiersz/P_11NettoZ', 'p_11NettoZ', 'Pozycja: wartość całkowita netto', null, 'net'],
			['//Fa/Platnosc/RachunekBankowy/NrRB', 'nrRB', 'Numer rachunku bankowego', 'bank_account', null],
			['//Podmiot1/Adres/KodKraju', 'kodKraju', 'Kod kraju sprzedającego', 'addresslevel1a', null],
			['//Podmiot2/Adres/KodKraju', 'kodKraju', 'Kod kraju kupującego', 'addresslevel1a:MultiCompany:companyid', null],
			['//Fa/RachunekBankowy', 'RachunekBankowy', 'Powiązany rachunek bankowy', null, null],
			['//Fa/RachunekBankowy/NazwaBanku', 'nazwaBanku', 'Nazwa banku', null, null],
			['//Fa/Adnotacje/P_16', 'P_16', 'Adnotacja „metoda kasowa"', 'cash_method', null],
			['yf_company', null, 'Organizacja', 'companyid', null],
			['//Stopka/Rejestry/REGON', 'regon', 'Stopka: REGON', null, null],
			['//Stopka/Rejestry/KRS', 'krs', 'Stopka: KRS', null, null],
			['//Podmiot2/DaneIdentyfikacyjne/Nazwa', 'nazwa', 'Nazwa kupującego', 'company_name:MultiCompany:companyid', null],
			['//Podmiot1/DaneIdentyfikacyjne/Nazwa', 'nazwa', 'Nazwa sprzedającego', 'company_name_a', null],
			['//Fa/DaneFaKorygowanej/NrFaKorygowanej', 'NrFaKorygowanej', 'Nr faktury korygowanej', null, null],
			['//Fa/DaneFaKorygowanej/NrKSeFFaKorygowanej', 'NrKSeFFaKorygowanej', 'Nr KSeF faktury korygowanej', null, null],
			['//Fa/TypKorekty', 'TypKorekty', 'Typ korekty', 'correction_type', null],
			['//Fa/PrzyczynaKorekty', 'PrzyczynaKorekty', 'Przyczyna korekty (opis)', 'description', null],
			['//Stopka/Rejestry/PelnaNazwa', 'stopka_pelna_nazwa', 'Stopka: pełna nazwa firmy', null, null],
			['//Fa/RodzajFaktury', 'rodzajFaktury', 'Rodzaj faktury', 'invoice_cost_type_ksef', null],
			['//Stopka/Informacje/StopkaFaktury', 'stopkaFaktury', 'Stopka: treść', 'attention', null],
			['//Fa/P_1', 'p_1', 'Data wystawienia', 'issue_time', null],
			['//Fa/DaneFaKorygowanej/DataWystFaKorygowanej', 'DataWystFaKorygowanej', 'Data wystawienia faktury korygowanej', 'issue_time', null],
			['yf_ksef_date', '', 'Data wystawienia faktury (KSeF)', 'ksef_date', null],
			['yf_ksef_number', null, 'Numer KSeF', 'ksef_number', null],
			['//Fa/Adnotacje/PMarzy', 'PMarzy', 'Procedura marży', 'margin_procedure', null],
			['//Fa/Platnosc/DataZaplaty', 'p_6Z', 'Data płatności', 'paymentdate', null],
			['//Fa/Platnosc/DataZaplatyCzesciowej', 'dataZaplatyCzesciowej', 'Data zapłaty częsciowej', 'paymentdate', null],
			['//Fa/Platnosc/TerminPlatnosci/Termin', 'Termin', 'Termin płatności', 'paymentdate', null],
			['//Fa/Platnosc/FormaPlatnosci', 'FormaPlatnosci', 'Forma płatności', 'payment_methods', null],
			['//Fa/Platnosc/LinkDoPlatnosci', 'LinkDoPlatnosci', 'Płatność: link do płatności', 'payment_url', null],
			['//Fa/Adnotacje/P_18', 'P_18', 'Adnotacja „odwrotne obciążenie"', 'reverse_charge', null],
			['//Fa/P_6', 'p_6', 'Data sprzedaży', 'saledate', null],
			['//Fa/FakturaZaliczkowa/NrFaZaliczkowej', 'NrFaZaliczkowej', 'Nr faktury zaliczkowej', null, null],
			['//Fa/FakturaZaliczkowa/NrKSeFFaZaliczkowej', 'NrKSeFFaZaliczkowej', 'Nr KSeF faktury zaliczkowej', null, null],
			['//Fa/Adnotacje/P_18A', 'P_18A', 'Adnotacja „mechanizm podzielonej płatności"', 'split_payment', null],
			['//Fa/P_2', 'p_2', 'Numer faktury', 'subject', null],
			['//Fa/P_15', 'p_15', 'Wartość całkowita', 'sum_gross', null],
			['//Fa/P_15Z', 'p_15Z', 'Wartość płatności', 'sum_gross', null],
			['//Fa/Platnosc/KwotaZaplatyCzesciowej', 'kwotaZaplatyCzesciowej', 'Kwota zapłaty częsciowej', 'sum_gross', null],
			['yf_net_sum', null, 'Suma netto', 'sum_total', null],
			['//Fa/Adnotacje/Zwolnienie/P_19A', 'P_19A', 'Powód zwolnienia podatku (ustawa)', 'tax_exemption_text', null],
			['//Fa/Adnotacje/Zwolnienie/P_19B', 'P_19B', 'Powód zwolnienia podatku B (dyrektywa)', 'tax_exemption_text', null],
			['//Fa/Adnotacje/Zwolnienie/P_19C', 'P_19C', 'Powód zwolnienia podatku (inne)', 'tax_exemption_text', null],
			['//Fa/Adnotacje/Zwolnienie/P_19', 'P_19', 'Rodzaj zwolnienia z podatku', 'tax_exemption_type', null],
			['//Podmiot2/DaneIdentyfikacyjne/NIP', 'nip', 'NIP kupującego', 'vat:MultiCompany:companyid', null],
			['//Podmiot1/DaneIdentyfikacyjne/NIP', 'nip', 'NIP sprzedającego', 'vat_id_a', null],
			['//Podmiot1/Adres/AdresL1', 'adresL1', 'Adres (linia 1) sprzedającego', 'full_address', null],
		];

		$isExists = (new Query())->from(CostInvoiceMapper::TABLENAME)->exists();
		if (!$isExists) {
			$dbCommand = Db::getInstance()->createCommand();
			$dbCommand->batchInsert(
				CostInvoiceMapper::TABLENAME,
				['key', 'ksef_field', 'name', 'field', 'inventory_field'],
				$mappingRows
			)->execute();
		}
	}
}
