<?php
/**
 * KSeF DownloadQueue Model class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Model\Client;

use N1ebieski\KSEFClient\ValueObjects\EncryptionKey;

/**
 * KSeF DownloadQueue Model class.
 */
final class DownloadQueue
{
	/** @var int|null Queue record ID */
	private ?int $id = null;

	/** @var int|null Settings ID */
	private ?int $settingsId = null;

	/** @var int|null Queue status */
	private ?int $status = null;

	/** @var string|null Import request reference number */
	private ?string $referenceNumber = null;

	/** @var string|null Date when request was created */
	private ?string $dateInput = null;

	/** @var string|null Date when request was last modified */
	private ?string $dateMod = null;

	/** @var string|null Key part for EncryptionKey */
	private ?string $key = null;

	/** @var string|null IV part for EncryptionKey */
	private ?string $iv = null;

	/** @var string|null Module name */
	private ?string $module = null;

	/**
	 * Get queue record ID.
	 *
	 * @return int|null
	 */
	public function getId(): ?int
	{
		return $this->id;
	}

	/**
	 * Set queue record ID.
	 *
	 * @param int|null $id
	 *
	 * @return self
	 */
	public function setId(?int $id): self
	{
		$this->id = $id;
		return $this;
	}

	/**
	 * Get settings ID.
	 *
	 * @return int|null
	 */
	public function getSettingsId(): ?int
	{
		return $this->settingsId;
	}

	/**
	 * Set settings ID.
	 *
	 * @param int|null $settingsId
	 *
	 * @return self
	 */
	public function setSettingsId(?int $settingsId): self
	{
		$this->settingsId = $settingsId;
		return $this;
	}

	/**
	 * Get queue status.
	 *
	 * @return int|null
	 */
	public function getStatus(): ?int
	{
		return $this->status;
	}

	/**
	 * Set queue status.
	 *
	 * @param int|null $status
	 *
	 * @return self
	 */
	public function setStatus(?int $status): self
	{
		$this->status = $status;
		return $this;
	}

	/**
	 * Get reference number.
	 *
	 * @return string|null
	 */
	public function getReferenceNumber(): ?string
	{
		return $this->referenceNumber;
	}

	/**
	 * Set reference number.
	 *
	 * @param string|null $referenceNumber
	 *
	 * @return self
	 */
	public function setReferenceNumber(?string $referenceNumber): self
	{
		$this->referenceNumber = $referenceNumber;
		return $this;
	}

	/**
	 * Get date input.
	 *
	 * @return string|null
	 */
	public function getDateInput(): ?string
	{
		return $this->dateInput;
	}

	/**
	 * Set date input.
	 *
	 * @param string|null $dateInput
	 *
	 * @return self
	 */
	public function setDateInput(?string $dateInput): self
	{
		$this->dateInput = $dateInput;
		return $this;
	}

	/**
	 * Get date modified.
	 *
	 * @return string|null
	 */
	public function getDateMod(): ?string
	{
		return $this->dateMod;
	}

	/**
	 * Set date modified.
	 *
	 * @param string|null $dateMod
	 *
	 * @return self
	 */
	public function setDateMod(?string $dateMod): self
	{
		$this->dateMod = $dateMod;
		return $this;
	}

	/**
	 * Get key part for EncryptionKey.
	 *
	 * @return string|null
	 */
	public function getKey(): ?string
	{
		return $this->key;
	}

	/**
	 * Set key part for EncryptionKey.
	 *
	 * @param string|null $key
	 *
	 * @return self
	 */
	public function setKey(?string $key): self
	{
		$this->key = $key;
		return $this;
	}

	/**
	 * Get IV part for EncryptionKey.
	 *
	 * @return string|null
	 */
	public function getIv(): ?string
	{
		return $this->iv;
	}

	/**
	 * Set IV part for EncryptionKey.
	 *
	 * @param string|null $iv
	 *
	 * @return self
	 */
	public function setIv(?string $iv): self
	{
		$this->iv = $iv;
		return $this;
	}
	
	/**
	 * Set module name.
	 *
	 * @param string|null $module
	 *
	 * @return self
	 */
	public function setModule(?string $module): self
	{
		$this->module = $module;
		return $this;
	}

	/**
	 * Get module name.
	 *
	 * @return string|null
	 */
	public function getModule(): ?string
	{
		return $this->module;
	}

	/**
	 * Get encryption key.
	 *
	 * @return EncryptionKey|null
	 */
	public function getEncryptionKey(): ?EncryptionKey
	{
		if (null === $this->key || null === $this->iv) {
			return null;
		}
		return EncryptionKey::from($this->key, $this->iv);
	}
}
