<?php
/**
 * KSeF identifier model class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Model\Connector;

use App\Integrations\KSeF\Exception\KSeFConfigurationException;
use N1ebieski\KSEFClient\ValueObjects\InternalId;
use N1ebieski\KSEFClient\ValueObjects\NIP;
use N1ebieski\KSEFClient\ValueObjects\NipVatUe;
use N1ebieski\KSEFClient\ValueObjects\PeppolId;

/**
 * KSeF identifier model class.
 */
class Identifier
{
	/** @var array Allowed identifier types. Based on N1ebieski\KSEFClient\ValueObjects */
	public const ALLOWED = [
		'NIP' => 'PLL_NIP',
		'NIP_VAT_UE' => 'PLL_NIP_VAT_UE',
		'INTERNAL_ID' => 'PLL_INTERNAL_ID',
		'PEPPOL_ID' => 'PLL_PEPPOL_ID',
	];

	/** @var string Identifier code. For example tax number. */
	private string $code;

	/** @var string Identifier type. Must be one of the allowed types. */
	private string $type;

	/** Identifier object for N1ebieski\KSEFClient. */
	private null|InternalId|NIP|NipVatUe|PeppolId $object = null;

	/**
	 * Constructor. Creates N1ebieski\KSEFClient identifier object from database data.
	 *
	 * @param string $code Identifier code. For example tax number.
	 * @param string $type Identifier type. Must be one of the allowed types. Check ALLOWED constant.
	 *
	 * @throws KSeFConfigurationException
	 */
	public function __construct(
		string $code,
		string $type,
	) {
		$this->code = $code;
		$this->type = $type;
		$this->objectFactory();
	}

	/**
	 * Get identifier code.
	 *
	 * @return string Identifier code. For example tax number.
	 */
	public function getCode(): string
	{
		return $this->code;
	}

	/**
	 * Get identifier type.
	 *
	 * @return string Identifier type.
	 */
	public function getType(): string
	{
		return $this->type;
	}

	/**
	 * Get identifier as N1ebieski\KSEFClient object.
	 *
	 * @return NIP|NipVatUe|InternalId|PeppolId|null
	 */
	public function asObject(): null|InternalId|NIP|NipVatUe|PeppolId
	{
		return $this->object;
	}

	/**
	 * Factory method to create N1ebieski\KSEFClient identifier object.
	 *
	 * @throws KSeFConfigurationException
	 */
	private function objectFactory(): void
	{
		if (!\in_array($this->type, array_keys(self::ALLOWED))) {
			throw new KSeFConfigurationException('ERR_KSEF_INVALID_IDENTIFIER_TYPE');
		}

		$this->object = match ($this->type) {
			'NIP_VAT_UE' => new NipVatUe($this->code),
			'INTERNAL_ID' => new InternalId($this->code),
			'PEPPOL_ID' => new PeppolId($this->code),
			default => new NIP($this->code),
		};
	}
}
