<?php
/**
 * KSeF Invoice Builder class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Builder;

use App\Integrations\KSeF\Service\Builder\Rules\FaAnnotationsRule;
use App\Integrations\KSeF\Service\Builder\Rules\FaCorrectionRule;
use App\Integrations\KSeF\Service\Builder\Rules\FaInvoiceTypeRule;
use App\Integrations\KSeF\Service\Builder\Rules\FaPartialAdvanceRule;
use App\Integrations\KSeF\Service\Builder\Rules\FaPaymentRule;
use App\Integrations\KSeF\Service\Builder\Rules\FaRowRule;
use App\Integrations\KSeF\Service\Builder\Rules\FaRule;
use App\Integrations\KSeF\Service\Builder\Rules\FaSettlementInvoiceRule;
use App\Integrations\KSeF\Service\Builder\Rules\FaSettlementRule;
use App\Integrations\KSeF\Service\Builder\Rules\FooterRule;
use App\Integrations\KSeF\Service\Builder\Rules\Subject1Rule;
use App\Integrations\KSeF\Service\Builder\Rules\Subject2Rule;
use App\Integrations\KSeF\Service\Mapper\MapperService;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\Fa;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\Faktura;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\Naglowek;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\Podmiot1;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\Podmiot2;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\Podmiot3;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\PodmiotUpowazniony;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\Stopka;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\Zalacznik;
use N1ebieski\KSEFClient\Support\Optional;

/**
 * KSeF Invoice Builder class.
 */
final class InvoiceBuilder extends AbstractBuilder
{
	/** @var Naglowek|null Header DTO */
	private ?Naglowek $header = null;

	/** @var Podmiot1|null Subject1 DTO */
	private ?Podmiot1 $subject1 = null;

	/** @var Podmiot2|null Subject2 DTO */
	private ?Podmiot2 $subject2 = null;

	/** @var Fa|null Fa DTO */
	private ?Fa $fa = null;

	/** @var Optional|Podmiot3[] Subject3 array */
	private array|Optional $subject3 = [];

	/** @var Optional|PodmiotUpowazniony[] AuthorizedSubject DTO */
	private Optional|PodmiotUpowazniony $authorizedSubject;

	/** @var Optional|Stopka[] Footer DTO */
	private Optional|Stopka $footer;

	/** @var Optional|Zalacznik[] Attachment DTO */
	private Optional|Zalacznik $attachment;

	/** {@inheritDoc} */
	public function __construct(MapperService $mapperService)
	{
		$this->authorizedSubject = new Optional();
		$this->footer = new Optional();
		$this->attachment = new Optional();
		$this->subject3 = new Optional();
		parent::__construct($mapperService);
	}

	/** {@inheritDoc} */
	public function build(): Faktura
	{
		$this->header = (new HeaderBuilder($this->mapperService))
			->build();

		$this->subject1 = (new Subject1Builder($this->mapperService))
			->addRule(new Subject1Rule())
			->fillFromRecord()
			->build();

		$this->subject2 = (new Subject2Builder($this->mapperService))
			->addRule(new Subject2Rule())
			->fillFromRecord()
			->build();

		$this->footer = (new FooterBuilder($this->mapperService))
			->addRule(new FooterRule())
			->fillFromRecord()
			->build();

		$this->fa = (new FaBuilder($this->mapperService))
			->addRule(new FaInvoiceTypeRule())
			->addRule(new FaAnnotationsRule())
			->addRule(new FaCorrectionRule())
			->addRule(new FaPartialAdvanceRule())
			->addRule(new FaPaymentRule())
			->addRule(new FaSettlementInvoiceRule())
			->addRule(new FaSettlementRule())
			->addRule(new FaRowRule())
			->addRule(new FaRule())
			->fillFromRecord()
			->build();

		return new Faktura(
			naglowek: $this->header,
			podmiot1: $this->subject1,
			podmiot2: $this->subject2,
			fa: $this->fa,
			podmiot3: empty($this->subject3) ? new Optional() : $this->subject3,
			podmiotUpowazniony: $this->authorizedSubject,
			stopka: $this->footer,
			zalacznik: $this->attachment,
		);
	}
}
