<?php

/**
 * KSeF Builder FaPaymentRule rule class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Builder\Rules;

use App\Integrations\KSeF\Model\Enum\Type;
use App\Integrations\KSeF\Service\Mapper\MapperService;
use App\Language;
use App\QueryGenerator;
use App\Record;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\FormaPlatnosciGroup;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\NrRBGroup;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\Platnosc;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\PlatnoscInnaGroup;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\RachunekBankowy;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\TerminPlatnosci;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\ZaplataCzesciowa;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\ZaplataCzesciowaGroup;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\ZaplataGroup;
use N1ebieski\KSEFClient\Support\Optional;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\DataZaplaty;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\DataZaplatyCzesciowej;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\FormaPlatnosci;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\KwotaZaplatyCzesciowej;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\NazwaBanku;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\NrRB;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\OpisPlatnosci;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\SWIFT;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\Termin;

/**
 * KSeF Builder FaPaymentRule rule class.
 */
class FaPaymentRule implements RuleInterface
{
	private ?MapperService $mapperService = null;
	private ?\Vtiger_Record_Model $recordModel = null;

	/** {@inheritDoc} */
	public function apply(array &$target, MapperService $mapperService): void
	{
		$this->mapperService = $mapperService;
		$this->recordModel = $mapperService->getRecord();

		$paymentDetails = $this->getPaymentDetails();
		$paymentDue = $this->getPaymentDue();
		$paymentForm = $this->getPaymentForm();
		$bankAccount = $this->getBankAccount();

		$paymentNode = new Platnosc(
			zaplataGroup: (!empty($paymentDetails)) ? $paymentDetails : new Optional(),
			terminPlatnosci: (!empty($paymentDue)) ? [$paymentDue] : [],
			platnoscGroup:  (!empty($paymentForm)) ? $paymentForm : new Optional(),
			rachunekBankowy: (!empty($bankAccount)) ? [$bankAccount] : [],
			rachunekBankowyFaktora: new Optional(), /* Not implemented yet */
			skonto: new Optional(), /* Not implemented yet */
		);

		$target['fa_platnosc'] = $paymentNode;
	}

	private function getPaymentDetails(): ZaplataCzesciowaGroup|ZaplataGroup|null
	{
		$paymentDetails = null;
		$paymentStatus = $this->mapperService->mapField('yf_payment_status', Type::STRING);
		$paymentDate = $this->mapperService->mapField('//Fa/Platnosc/TerminPlatnosci/Termin', Type::DATE);
		$paymentSum = $this->mapperService->mapField('yf_payment_sum', Type::FLOAT);
		$invoiceTotal = $this->mapperService->mapField('//Fa/P_15', Type::FLOAT);

		if ('PLL_PAID' === $paymentStatus && !empty($paymentDate)) {
			$paymentDetails = new ZaplataGroup(dataZaplaty: new DataZaplaty($paymentDate));
		} elseif ($paymentSum > 0.00 && $paymentSum < $invoiceTotal) {
			$relatedPayments = $this->getRelatedPayments($this->recordModel->getId());
			$partialPayments = [];
			if (!empty($relatedPayments)) {
				foreach ($relatedPayments as $payment) {
					$value = $this->mapperService->mapField('//Fa/Platnosc/KwotaZaplatyCzesciowej', Type::FLOAT, $payment);
					$date = $this->mapperService->mapField('//Fa/Platnosc/DataZaplatyCzesciowej', Type::DATE, $payment);

					if (!empty($value) && !empty($date)) {
						new ZaplataCzesciowa(
							kwotaZaplatyCzesciowej: new KwotaZaplatyCzesciowej($value),
							dataZaplatyCzesciowej: new DataZaplatyCzesciowej($date),
							platnoscGroup: new Optional() /* Not implemented yet */
						);
					}
				}
			}

			$paymentDetails = new ZaplataCzesciowaGroup(zaplataCzesciowa: $partialPayments);
		}

		return $paymentDetails;
	}

	/**
	 * @return TerminPlatnosci|null
	 */
	private function getPaymentDue(): ?TerminPlatnosci
	{
		$paymentDue = null;
		$paymentDate = $this->mapperService->mapField('//Fa/Platnosc/TerminPlatnosci/Termin', Type::DATE);

		if (!empty($paymentDate)) {
			$paymentDue = new TerminPlatnosci(
				termin: new Termin($paymentDate),
				terminOpis: new Optional(), /* Not implemented yet */
			);
		}

		return $paymentDue;
	}

	/**
	 * @return FormaPlatnosciGroup|PlatnoscInnaGroup|null
	 */
	private function getPaymentForm(): FormaPlatnosciGroup|PlatnoscInnaGroup|null
	{
		return match ($this->mapperService->mapField('//Fa/Platnosc/FormaPlatnosci', Type::STRING)) {
			'PLL_TRANSFER' => new FormaPlatnosciGroup(FormaPlatnosci::Przelew),
			'PLL_WIRE_TRANSFER' => new FormaPlatnosciGroup(FormaPlatnosci::Przelew),
			'PLL_CASH' => new FormaPlatnosciGroup(FormaPlatnosci::Gotowka),
			'PLL_CHECK' => new FormaPlatnosciGroup(FormaPlatnosci::Czek),
			'PLL_CARD' => new FormaPlatnosciGroup(FormaPlatnosci::Karta),
			'PLL_VOUCHER' => new FormaPlatnosciGroup(FormaPlatnosci::Bon),
			'PLL_MOBILE' => new FormaPlatnosciGroup(FormaPlatnosci::Mobilna),
			'PLL_LOAN' => new FormaPlatnosciGroup(FormaPlatnosci::Kredyt),

			'PLL_CASH_ON_DELIVERY' => new PlatnoscInnaGroup(new OpisPlatnosci(Language::translate('PLL_CASH_ON_DELIVERY', $this->recordModel->getModuleName()))),
			'PLL_REDSYS' => new PlatnoscInnaGroup(new OpisPlatnosci(Language::translate('PLL_REDSYS', $this->recordModel->getModuleName()))),
			'PLL_DOTPAY' => new PlatnoscInnaGroup(new OpisPlatnosci(Language::translate('PLL_DOTPAY', $this->recordModel->getModuleName()))),
			'PLL_PAYPAL' => new PlatnoscInnaGroup(new OpisPlatnosci(Language::translate('PLL_PAYPAL', $this->recordModel->getModuleName()))),
			'PLL_PAYPAL_EXPRESS' => new PlatnoscInnaGroup(new OpisPlatnosci(Language::translate('PLL_PAYPAL_EXPRESS', $this->recordModel->getModuleName()))),

			default => null,
		};
	}

	/**
	 * @return \Vtiger_Record_Model|null
	 */
	private function getBankAccount(): ?RachunekBankowy
	{
		$bankAccount = null;

		$bankAccountId = $this->mapperService->mapField('//Fa/RachunekBankowy', Type::INTEGER);
		if (!empty($bankAccountId) && Record::isExists($bankAccountId)) {
			$number = $this->mapperService->mapField('//Fa/RachunekBankowy/NrRB', Type::STRING);
			$swift = $this->mapperService->mapField('//Fa/RachunekBankowy/SWIFT', Type::STRING);
			$bankName = $this->mapperService->mapField('//Fa/RachunekBankowy/NazwaBanku', Type::STRING);
			$bankAccount = new RachunekBankowy(
				nrRBGroup: new NrRBGroup(
					nrRB: new NrRB($number),
					swift: (!empty($swift)) ? new SWIFT($swift) : new Optional(),
				),
				rachunekWlasnyBanku: new Optional(), /* Not implemented yet */
				nazwaBanku: (!empty($bankName)) ? new NazwaBanku($bankName) : new Optional(),
				opisRachunku: new Optional(), /* Not implemented yet */
			);
		}

		return $bankAccount;
	}

	/**
	 * @param int $recordId
	 *
	 * @return \Vtiger_Record_Model[]
	 */
	private function getRelatedPayments(int $recordId): array
	{
		$query = new QueryGenerator('PaymentsIn');
		$query->permissions = false;
		$query->setFields(['paymentsinid']);
		$query->addCondition('finvoiceid', $recordId, 'e');
		$query->addCondition('paymentsin_status', 'PLL_PAID', 'e');

		$results = [];
		foreach ($query->createQuery()->all() as $item) {
			$results[] = \Vtiger_Record_Model::getInstanceById($item['paymentsinid']);
		}

		return $results;
	}
}
