<?php

/**
 * KSeF Builder FaRule rule class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Builder\Rules;

use App\Integrations\KSeF\Model\Enum\Type;
use App\Integrations\KSeF\Service\Mapper\MapperService;
use App\TextUtils;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\DodatkowyOpis;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\FaWiersz;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\P_PMarzyGroup;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\Klucz;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_12;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\P_18;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\RodzajFaktury;
use N1ebieski\KSEFClient\ValueObjects\Requests\Sessions\Wartosc;

/**
 * KSeF Builder FaRule rule class.
 */
class FaRule implements RuleInterface
{
	/** {@inheritDoc} */
	public function apply(array &$target, MapperService $mapperService): void
	{
		$inventoryData = $mapperService->getRecord()->getInventoryData();
		$summaryValues = $this->getSummaryValues($inventoryData);
		$target['//Fa/KodWaluty'] = $mapperService->mapField('//Fa/KodWaluty', Type::STRING, current($inventoryData)) ?: 'PLN';

		$target['//Fa/P_1'] = $mapperService->mapField('//Fa/P_1', Type::DATE);
		$target['//Fa/P_1M'] = $mapperService->mapField('//Fa/P_1M', Type::STRING);
		$target['//Fa/P_2'] = $mapperService->mapField('//Fa/P_2', Type::STRING);

		$target['//Fa/P_15'] = $mapperService->mapField('//Fa/P_15', Type::FLOAT);

		$target['//Fa/P_6'] = $mapperService->mapField('//Fa/P_6', Type::DATE);
		$target['okresFa']['//Fa/P_6_Od'] = $mapperService->mapField('//Fa/P_6_Od', Type::DATE);
		$target['okresFa']['//Fa/P_6_Do'] = $mapperService->mapField('//Fa/P_6_Do', Type::DATE);

		$target['//Fa/P_13_1'] = $summaryValues[23]['net'] ?? 0.00;
		$target['//Fa/P_14_1'] = $summaryValues[23]['tax'] ?? 0.00;
		$target['//Fa/P_14_1w'] = $summaryValues[23]['tax_pl'] ?? 0.00;

		$target['//Fa/P_13_2'] = $summaryValues[8]['net'] ?? 0.00;
		$target['//Fa/P_14_2'] = $summaryValues[8]['tax'] ?? 0.00;
		$target['//Fa/P_14_2w'] = $summaryValues[8]['tax_pl'] ?? 0.00;

		$target['//Fa/P_13_3'] = $summaryValues[5]['net'] ?? 0.00;
		$target['//Fa/P_14_3'] = $summaryValues[5]['tax'] ?? 0.00;
		$target['//Fa/P_14_3w'] = $summaryValues[5]['tax_pl'] ?? 0.00;

		$target['//Fa/P_13_6_1'] = 0.00;
		$target['//Fa/P_13_6_2'] = 0.00;
		$target['//Fa/P_13_6_3'] = 0.00;
		switch ($mapperService->mapField('yf_finvoice_type', Type::STRING)) {
			case 'PLL_DOMESTIC_INVOICE':
				$target['//Fa/P_13_6_1'] = $summaryValues[0]['net'] ?? 0.00;
				break;
			case 'PLL_IC_INVOICE':
				$target['//Fa/P_13_6_2'] = $summaryValues[0]['net'] ?? 0.00;
				break;
			case 'PLL_FOREIGN_INVOICE':
				$target['//Fa/P_13_6_3'] = $summaryValues[0]['net'] ?? 0.00;
				break;
		}

		$target['//Fa/P_13_7'] = ('PLL_DOMESTIC_INVOICE' === $mapperService->mapField('yf_finvoice_type', Type::STRING))
			? $this->getTotalExempt($target)
			: 0.00;

		$target['//Fa/P_13_8'] = ('PLL_FOREIGN_INVOICE' === $mapperService->mapField('yf_finvoice_type', Type::STRING))
			? $target['//Fa/P_15']
			: 0.00;

		$target['//Fa/P_13_9'] = ('PLL_IC_INVOICE' === $mapperService->mapField('yf_finvoice_type', Type::STRING))
			? $target['//Fa/P_15']
			: 0.00;

		$target['//Fa/P_13_10'] = (P_18::OdwrotneObciazenie === $target['fa_adnotacje']?->p_18)
			? $target['//Fa/P_15']
			: 0.00;

		$target['//Fa/P_13_11'] = ($target['fa_adnotacje']?->pMarzy?->p_PMarzyGroup instanceof P_PMarzyGroup)
			? $target['//Fa/P_15']
			: 0.00;

		if (RodzajFaktury::Zal === $target['//Fa/RodzajFaktury']) {
			$target['fa_kursWalutyZ'] = $mapperService->mapField('//Fa/KursWalutyZ', Type::FLOAT, current($inventoryData));
		}
	}

	private function getSummaryValues(array $inventoryData): array
	{
		$summary = [];
		foreach ($inventoryData as $item) {
			$taxPercent = (int) ($item['tax_percent'] ?? 0);
			if (!isset($summary[$taxPercent])) {
				$summary[$taxPercent] = ['net' => 0.0, 'gross' => 0.0, 'tax' => 0.0, 'tax_pl' => 0.0];
			}
			$summary[$taxPercent]['net'] += (float) ($item['net'] ?? 0);
			$summary[$taxPercent]['gross'] += (float) ($item['gross'] ?? 0);
			$summary[$taxPercent]['tax'] += (float) ($item['tax'] ?? 0);

			$currencyRate = $this->getCurrencyRate($item['currency'] ?? null, $item['currencyparam'] ?? null);
			$taxPl = (float) ($item['tax_pl'] ?? $item['tax'] ?? 0);
			$summary[$taxPercent]['tax_pl'] += round($taxPl * $currencyRate, 2);
		}

		return $summary;
	}

	private function getCurrencyRate(?int $currency, ?string $currencyParam): float
	{
		$currencyRate = 1.00;

		if (!empty($currencyParam) && !empty($currency)) {
			$currencyParamDecoded = json_decode($currencyParam, true);
			if (isset($currencyParamDecoded[$currency]['value'])) {
				$currencyRate = (float) $currencyParamDecoded[$currency]['value'];
			}
		}
		return $currencyRate;
	}

	private function getTotalExempt(array $target): float
	{
		$total = 0.00;

		/** @var FaWiersz $row */
		foreach ($target['faWiersz'] as $row) {
			if (P_12::Zw === $row->p_12) {
				$total += $row->p_11->value;
			}
		}

		return $total;
	}
}
