<?php

/**
 * KSeF Invoice Service class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Invoice;

use App\Integrations\KSeF\KSeF;
use App\Integrations\KSeF\Model\Enum\Mode;
use App\Integrations\KSeF\Model\Enum\Status;
use App\Integrations\KSeF\Repository\CertificateRepository;
use App\Integrations\KSeF\Repository\QueueRepository;
use App\Integrations\KSeF\Service\Builder\InvoiceBuilder;
use App\Integrations\KSeF\Service\Mapper\MapperService;
use N1ebieski\KSEFClient\DTOs\Requests\Sessions\Faktura;

/**
 * KSeF Invoice Service class.
 */
final class InvoiceService
{
	/** @var \Vtiger_Record_Model Invoice record model instance */
	private \Vtiger_Record_Model $invoiceRecord;

	/** @var MapperService Mapper service for field mapping operations */
	private MapperService $mapperService;

	/** @var InvoiceBuilder Invoice builder for constructing KSeF invoice DTOs */
	private InvoiceBuilder $invoiceBuilder;

	/** @var Faktura|null Built KSeF invoice DTO */
	private ?Faktura $ksefInvoice = null;

	/**
	 * Constructor.
	 *
	 * @param \Vtiger_Record_Model $invoiceRecordModel
	 * @param MapperService|null   $mapperService      Optional mapper service (for testing)
	 */
	public function __construct(\Vtiger_Record_Model $invoiceRecordModel, ?MapperService $mapperService = null)
	{
		$this->invoiceRecord = $invoiceRecordModel;
		$this->mapperService = $mapperService ?? new MapperService();
	}

	/**
	 * Build and return KSeF invoice DTO.
	 *
	 * @return Faktura
	 */
	public function makeInvoice(): Faktura
	{
		if (!$this->mapperService->isInitialized()) {
			$this->mapperService->buildMapper($this->invoiceRecord);
		}
		$this->invoiceBuilder = new InvoiceBuilder($this->mapperService);
		$this->ksefInvoice = $this->invoiceBuilder->build();

		return $this->ksefInvoice;
	}

	/**
	 * Check current KSeF status of invoice.
	 *
	 * @return bool true if invoice can be sent to KSeF, false otherwise
	 */
	public function isAllowedToSend(): bool
	{
		return 'PLL_ACCEPTED' === $this->invoiceRecord->get('finvoice_status')
				&& 1 !== $this->invoiceRecord->get('ksef_disable')
				&& !$this->invoiceRecord->get('ksef_number')
				&& $this->invoiceRecord->get('companyid')
				&& (new CertificateRepository())->findByCompanyId($this->invoiceRecord->get('companyid'))
				&& !(new QueueRepository())->exists($this->invoiceRecord->getId());
	}

	public function isAllowedToReSend(): bool
	{
		return 'PLL_ACCEPTED' === $this->invoiceRecord->get('finvoice_status')
				&& 1 !== $this->invoiceRecord->get('ksef_disable')
				&& !$this->invoiceRecord->get('ksef_number')
				&& $this->invoiceRecord->get('companyid')
				&& (new CertificateRepository())->findByCompanyId($this->invoiceRecord->get('companyid'))
				&& Status::ERROR->value === (new QueueRepository())->getLastStatus($this->invoiceRecord->getId());
	}

	/**
	 * Update invoice status with data received from KSeF.
	 *
	 * @param string $sessionKey
	 * @param int    $ordinalNumber
	 * @param string|null $ksefNumber
	 * @param string|null $invoiceKey
	 * @param string|null $acquisitionDate
	 * @param string|null $invoicingMode
	 * @param object|null $status
	 *
	 * @return void
	 */
	public static function updateInvoiceKSeFStatus(
		string $sessionKey,
		int $ordinalNumber,
		?string $ksefNumber,
		?string $invoiceKey,
		?string $acquisitionDate,
		?string $invoicingMode,
		?object $status,
	): void {
		$queueRepository = new QueueRepository();
		$queueObject = $queueRepository->findOneBySession($sessionKey, $ordinalNumber);

		if (200 === $status?->code) {
			$queueObject
				->setStatus(Status::DONE->value)
				->setKsefNumber($ksefNumber)
				->setInvoiceKey($invoiceKey)
				->setDateAcquisition($acquisitionDate ? (new \DateTime($acquisitionDate))->format('Y-m-d H:i:s') : null)
				->setMode($invoicingMode ? Mode::fromString($invoicingMode) : null);
		} else {
			KSeF::getLogger()->error(
				'Invoice processing did not complete successfully.',
				[
					'params' => [
						'record' => $queueObject?->getInvoiceRecordId(),
						'KSeFNumber' => $ksefNumber,
						'invoiceKey' => $invoiceKey,
						'ordinalNumber' => $ordinalNumber,
						'responseCode' => $status?->code,
						'responseDesc' => $status?->description,
					],
				]
			);
			$queueObject->setStatus(Status::ERROR->value);
		}
		$queueRepository->update($queueObject);
	}
}
