<?php

/**
 * KSeF Mapper Service class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Mapper;

use App\Db\Query;
use App\Integrations\KSeF\Exception\KSeFMapperException;
use App\Integrations\KSeF\Model\Enum\Type;
use App\Integrations\KSeF\Model\Mapper\AbstractMapper;
use App\Integrations\KSeF\Model\Mapper\CorrectingInvoiceMapper;
use App\Integrations\KSeF\Model\Mapper\CostInvoiceMapper;
use App\Integrations\KSeF\Model\Mapper\Pair;
use App\Integrations\KSeF\Model\Mapper\SalesInvoiceMapper;
use App\Integrations\KSeF\Service\Mapper\Resolver\AddressValueResolver;
use App\Integrations\KSeF\Service\Mapper\Resolver\InventoryValueResolver;
use App\Integrations\KSeF\Service\Mapper\Resolver\RecordValueResolver;
use App\Integrations\KSeF\Service\Mapper\Resolver\RelatedRecordValueResolver;
use App\Integrations\KSeF\Service\Mapper\Resolver\ValueResolverRegistry;

/**
 * KSeF Mapper Service class.
 */
final class MapperService
{
	/**
	 * @var AbstractMapper|null Current mapper instance
	 */
	private ?AbstractMapper $mapper = null;

	private ?ValueResolverRegistry $resolverRegistry = null;

	/**
	 * Check if mapper is initialized.
	 *
	 * @return bool
	 */
	public function isInitialized(): bool
	{
		return null !== $this->mapper;
	}

	/**
	 * Map field value with type casting.
	 *
	 * @param string                          $key
	 * @param Type                            $type
	 * @param \Vtiger_Record_Model|array|null $data Optional record or inventory row data, if null uses default record model
	 *
	 * @throws KSeFMapperException
	 *
	 * @return mixed
	 */
	public function mapField(string $key, Type $type, array|\Vtiger_Record_Model|null $data = null): mixed
	{
		if (!$this->mapper) {
			throw new KSeFMapperException('Mapper not initialized. Call buildMapper() first.');
		}

		$pair = $this->mapper->get($key);

		if (!$pair) {
			throw new KSeFMapperException("Mapping pair for key '{$key}' not found in mapper: " . $this->mapper->getModule());
		}

		$registry = $this->getResolverRegistry();

		try {
			$value = $registry->resolve($pair, $data);
		} catch (\Throwable $t) {
			throw new KSeFMapperException($t->getMessage() . "|field: {$key}|record: {$this->getRecord()->getId()}", 0, $t);
		}

		return $type->cast($value);
	}

	/**
	 * Load pairs from database into mapper.
	 *
	 * @return void
	 */
	public function loadPairs(): void
	{
		$rows = (new Query())
			->from($this->mapper::TABLENAME)
			->all();

		foreach ($rows as $row) {
			$pair = (new Pair())
				->setKey($row['key'])
				->setKsefField($row['ksef_field'])
				->setName($row['name'])
				->setInventoryField($row['inventory_field'])
				->setField($row['field']);

			$this->mapper->addPair($pair);
		}
	}

	public function getMapper(): ?AbstractMapper
	{
		return $this->mapper;
	}

	/**
	 * Load mapper for XML reading (without record).
	 *
	 * @param string $moduleName Module name (FInvoice, FInvoiceCost, FInvoiceProforma)
	 *
	 * @throws KSeFMapperException
	 *
	 * @return self
	 */
	public function loadMapper(string $moduleName): self
	{
		$this->mapper = match ($moduleName) {
			'FInvoice' => new SalesInvoiceMapper(),
			'FInvoiceProforma' => new CorrectingInvoiceMapper(),
			'FInvoiceCost' => new CostInvoiceMapper(),
			default => throw new KSeFMapperException("Unsupported module: {$moduleName}"),
		};

		$this->loadPairs();

		return $this;
	}

	/**
	 * Mapper factory method for XML writing (with record).
	 *
	 * @param \Vtiger_Record_Model $record
	 *
	 * @throws KSeFMapperException
	 *
	 * @return self
	 */
	public function buildMapper(\Vtiger_Record_Model $record): self
	{
		$this->mapper = match ($record->getModuleName()) {
			SalesInvoiceMapper::MODULE => new SalesInvoiceMapper(),
			CorrectingInvoiceMapper::MODULE => new CorrectingInvoiceMapper(),
			CostInvoiceMapper::MODULE => new CostInvoiceMapper(),

			default => throw new KSeFMapperException('Unsupported module.'),
		};

		$this->loadPairs();
		$this->mapper->setRecord($record);

		return $this;
	}

	/**
	 * Get module name from current mapper.
	 *
	 * @throws KSeFMapperException
	 *
	 * @return string
	 */
	public function getModuleName(): string
	{
		if (!$this->mapper) {
			throw new KSeFMapperException('Mapper not initialized. Call buildMapper() first.');
		}

		return $this->mapper::MODULE;
	}

	/**
	 * Get the invoice record model.
	 *
	 * @throws KSeFMapperException
	 *
	 * @return \Vtiger_Record_Model
	 */
	public function getRecord(): \Vtiger_Record_Model
	{
		if (!$this->mapper) {
			throw new KSeFMapperException('Mapper not initialized. Call buildMapper() first.');
		}

		return $this->mapper->getRecord();
	}

	private function getResolverRegistry(): ValueResolverRegistry
	{
		if (!$this->resolverRegistry) {
			$this->resolverRegistry = new ValueResolverRegistry();

			$this->resolverRegistry
				->register(new InventoryValueResolver())
				->register(new AddressValueResolver())
				->register(new RelatedRecordValueResolver())
				->register(new RecordValueResolver())
				->setMapper($this->mapper);
		}
		return $this->resolverRegistry;
	}
}
