<?php

/**
 * KSeF Inventory Value Resolver class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Mapper\Resolver;

use App\Fields\Currency;
use App\Integrations\KSeF\Exception\KSeFMapperException;
use App\Integrations\KSeF\Model\Mapper\Pair;
use App\Json;

class InventoryValueResolver implements ValueResolverInterface
{
	public function supports(Pair $pair, array|\Vtiger_Record_Model|null $data): bool
	{
		return \is_array($data);
	}

	public function resolve(
		Pair $pair,
		array|\Vtiger_Record_Model|null $data,
		ValueResolverRegistry $registry
	): mixed {
		$moduleName = $registry->getMapper()->getRecord()->getModuleName();
		$inventory = \Vtiger_Inventory_Model::getInstance($moduleName);

		if (($fn = $pair->getField()) && method_exists($this, $fn)) {
			return $this->{$fn}($data);
		}

		$fieldModel = $inventory->getField($pair->getInventoryField());
		if (!$fieldModel) {
			return $data[$pair->getInventoryField()] ?? null;
		}

		$value = $data[$fieldModel->getName()] ?? null;

		return match ($fieldModel->getType()) {
			'Currency' => $value ? Currency::getById((int) $value)['currency_code'] : 'PLN',
			'Name' => $fieldModel->getDisplayValue($value, $data, true),
			default => $value,
		};
	}

	/**
	 * Get currency rate from inventory data.
	 *
	 * @param array $data
	 *
	 * @return float|null Null if currency is PLN or not set
	 */
	private function getCurrencyRate(array $data): ?float
	{
		$currencyRate = null;
		$currencyId = $data['currency'] ?? null;
		if ($currencyId && 'PLN' !== (Currency::getById($currencyId)['currency_code'] ?? null)) {
			$currencyParams = $inventory['currencyparam'] ?? '[]';
			$currencyRate = Json::decode($currencyParams)[$currencyId]['value'] ?? null;
		}

		return $currencyRate;
	}

	private function getTaxPercent(array $data)
	{
		$taxPercent = $data['tax_percent'] ?? null;
		if (null === $taxPercent) {
			$taxParam = Json::decode($data['taxparam'] ?? '[]') ?: [];
			if ($taxParam) {
				$type = $taxParam['aggregationType'];
				if (\is_array($type)) {
					throw new KSeFMapperException('Not support multi aggregation in TAX.');
				}
				$taxPercent = (float) $taxParam["{$type}Tax"];
			}
		}

		return $taxPercent;
	}
}
