<?php

/**
 * KSeF Abstract Record Builder class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Reader\RecordBuilder;

use App\Integrations\KSeF\Exception\KSeFXmlException;
use App\Integrations\KSeF\Service\Mapper\MapperService;
use App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules\RuleInterface;

/**
 * KSeF Abstract Record Builder class.
 *
 * Base class for building Vtiger records from KSeF XML using rule-based architecture.
 */
abstract class AbstractRecordBuilder
{
	/** @var ?MapperService Mapper service for field mapping operations */
	protected ?MapperService $mapperService = null;

	/** @var RuleInterface[] Array of rules to apply during building */
	protected array $rules = [];

	/** @var array Context data passed to rules */
	protected array $context = [];

	/** @var \SimpleXMLElement|null The XML element being processed */
	protected ?\SimpleXMLElement $xml = null;

	protected int $companyId;

	/**
	 * Add a rule to the builder.
	 *
	 * @param RuleInterface $rule The rule to add
	 *
	 * @return self
	 */
	public function addRule(RuleInterface $rule): self
	{
		$this->rules[] = $rule;

		return $this;
	}

	/**
	 * Get the XML element being processed.
	 *
	 * @return \SimpleXMLElement|null
	 */
	public function getXml(): ?\SimpleXMLElement
	{
		return $this->xml;
	}

	/**
	 * Get the mapper service.
	 *
	 * @return ?MapperService
	 */
	public function getMapperService(): ?MapperService
	{
		return $this->mapperService;
	}

	/**
	 * Get the array of rules.
	 *
	 * @return RuleInterface[]
	 */
	public function getRules(): array
	{
		return $this->rules;
	}

	/**
	 * Get optional string value from XML using XPath.
	 *
	 * @param string                 $path The XPath expression to query
	 * @param \SimpleXMLElement|null $xml  Optional XML element to query, uses $this->xml if null
	 *
	 * @return string|null The string value if found, null otherwise
	 */
	public function getOptionalString(string $path, ?\SimpleXMLElement $xml = null): ?string
	{
		$xml = (null === $xml) ? $this->xml : $xml;
		$value = $xml->xpath($path);

		return !empty($value) ? (string) $value[0] : null;
	}

	/**
	 * Get required string value from XML using XPath.
	 *
	 * @param string                 $path The XPath expression to query
	 * @param \SimpleXMLElement|null $xml  Optional XML element to query, uses $this->xml if null
	 *
	 * @throws KSeFXmlException If the value is not found
	 *
	 * @return string The string value
	 */
	public function getRequiredString(string $path, ?\SimpleXMLElement $xml = null): string
	{
		$value = $this->getOptionalString($path, $xml);
		if (null === $value) {
			throw new KSeFXmlException("Required field missing: {$path}");
		}

		return $value;
	}

	/**
	 * Get optional decimal (float) value from XML using XPath.
	 *
	 * @param string                 $path The XPath expression to query
	 * @param \SimpleXMLElement|null $xml  Optional XML element to query, uses $this->xml if null
	 *
	 * @return float|null The float value if found, null otherwise
	 */
	public function getOptionalDecimal(string $path, ?\SimpleXMLElement $xml = null): ?float
	{
		$value = $this->getOptionalString($path, $xml);

		return null !== $value ? (float) $value : null;
	}

	/**
	 * Get required decimal (float) value from XML using XPath.
	 *
	 * @param string                 $path The XPath expression to query
	 * @param \SimpleXMLElement|null $xml  Optional XML element to query, uses $this->xml if null
	 *
	 * @throws KSeFXmlException If the value is not found
	 *
	 * @return float The float value
	 */
	public function getRequiredDecimal(string $path, ?\SimpleXMLElement $xml = null): float
	{
		$value = $this->getOptionalString($path, $xml);
		if (null === $value) {
			throw new KSeFXmlException("Required field missing: {$path}");
		}

		return (float) $value;
	}

	/**
	 * Get optional date value from XML using XPath.
	 *
	 * @param string                 $path The XPath expression to query
	 * @param \SimpleXMLElement|null $xml  Optional XML element to query, uses $this->xml if null
	 *
	 * @throws KSeFXmlException If date format is invalid
	 *
	 * @return string|null The date in Y-m-d format if found, null otherwise
	 */
	public function getOptionalDate(string $path, ?\SimpleXMLElement $xml = null): ?string
	{
		$value = $this->getOptionalString($path, $xml);
		if (null === $value) {
			return null;
		}

		$date = \DateTime::createFromFormat('Y-m-d', $value);

		if (false === $date) {
			throw new KSeFXmlException("Invalid date format for field: {$path}.");
		}

		return $date->format('Y-m-d');
	}

	/**
	 * Get required date value from XML using XPath.
	 *
	 * @param string                 $path The XPath expression to query
	 * @param \SimpleXMLElement|null $xml  Optional XML element to query, uses $this->xml if null
	 *
	 * @throws KSeFXmlException If the value is not found or date format is invalid
	 *
	 * @return string The date in Y-m-d format
	 */
	public function getRequiredDate(string $path, ?\SimpleXMLElement $xml = null): string
	{
		$value = $this->getOptionalString($path, $xml);
		if (null === $value) {
			throw new KSeFXmlException("Required field missing: {$path}");
		}

		$date = \DateTime::createFromFormat('Y-m-d', $value);

		if (false === $date) {
			throw new KSeFXmlException("Invalid date format for field: {$path}.");
		}

		return $date->format('Y-m-d');
	}

	/**
	 * Get optional boolean value from XML using XPath.
	 *
	 * @param string                 $path The XPath expression to query
	 * @param \SimpleXMLElement|null $xml  Optional XML element to query, uses $this->xml if null
	 *
	 * @return bool|null The boolean value (true if '1', false otherwise) if found, null if not found
	 */
	public function getOptionalBool(string $path, ?\SimpleXMLElement $xml = null): ?bool
	{
		$value = $this->getOptionalString($path, $xml);

		return null !== $value ? '1' === $value : null;
	}

	/**
	 * Get required boolean value from XML using XPath.
	 *
	 * @param string                 $path The XPath expression to query
	 * @param \SimpleXMLElement|null $xml  Optional XML element to query, uses $this->xml if null
	 *
	 * @throws KSeFXmlException If the value is not found
	 *
	 * @return bool The boolean value (true if '1', false otherwise)
	 */
	public function getRequiredBool(string $path, ?\SimpleXMLElement $xml = null): bool
	{
		$value = $this->getOptionalString($path, $xml);
		if (null === $value) {
			throw new KSeFXmlException("Required field missing: {$path}");
		}

		return '1' === $value;
	}

	/**
	 * Set field value on record if mapping exists for the given XPath key.
	 *
	 * Uses the internal MapperService to resolve the XPath key to a field name.
	 * If a mapping exists, sets the field value on the record.
	 *
	 * @param \Vtiger_Record_Model $record The record to set field on
	 * @param string               $key    The XML XPath key to map (e.g., '//Fa/P_2')
	 * @param int|string           $value  The value to set
	 *
	 * @return void
	 */
	public function setFieldIfMapped(
		\Vtiger_Record_Model $record,
		string $key,
		int|string $value,
	): void {
		$yfFieldName = $this->mapperService->getMapper()->get($key)?->getField();
		if ($yfFieldName) {
			$record->set($yfFieldName, $value);
		}
	}

	public function getCompanyId(): int
	{
		return $this->companyId;
	}

	/**
	 * Apply all rules to the record.
	 *
	 * @param \SimpleXMLElement    $xml    The XML element to read from
	 * @param \Vtiger_Record_Model $record The record model to populate
	 *
	 * @return void
	 */
	protected function applyRules(\SimpleXMLElement $xml, \Vtiger_Record_Model $record): void
	{
		$this->xml = $xml;
		foreach ($this->rules as $rule) {
			$rule->apply($record, $this->context);
		}
	}
}
