<?php
/**
 * KSeF Reader Annotations Rule class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules;

use App\Integrations\KSeF\Service\Reader\RecordBuilder\AbstractRecordBuilder;

/**
 * KSeF Reader Annotations Rule class.
 *
 * Handles building annotation fields from //Fa/Adnotacje section:
 * - P_16, P_17, P_18 (boolean annotations)
 * - P_18A (split payment indicator)
 * - Zwolnienie/P_19, P_19A, P_19B, P_19C (tax exemption)
 *
 * Not implemented: NoweSrodkiTransportu (new means of transport).
 */
final class AnnotationsRule implements RuleInterface
{
	private AbstractRecordBuilder $builder;

	/** {@inheritDoc} */
	public function __construct(AbstractRecordBuilder $builder)
	{
		$this->builder = $builder;
	}

	/** {@inheritDoc} */
	public function apply(
		\Vtiger_Record_Model $record,
		array $context = [],
	): void {
		$annotations = [];

		$p16 = $this->builder->getOptionalBool('//Fa/Adnotacje/P_16');
		if (true === $p16) {
			$annotations[] = 'P_16';
		}

		$p17 = $this->builder->getOptionalBool('//Fa/Adnotacje/P_17');
		if (true === $p17) {
			$annotations[] = 'P_17';
		}

		$p18 = $this->builder->getOptionalBool('//Fa/Adnotacje/P_18');
		if (true === $p18) {
			$annotations[] = 'P_18';
		}

		$p18a = $this->builder->getOptionalString('//Fa/Adnotacje/P_18A');
		if ($p18a) {
			$splitPaymentValue = ('1' === $p18a) ? '1' : '0';
			$this->builder->setFieldIfMapped($record, '//Fa/Adnotacje/P_18A', $splitPaymentValue);
		}

		$this->applyTaxExemption($record);

		// Not implemented yet: NoweSrodkiTransportu - (//Fa/Adnotacje/NoweSrodkiTransportu)
	}

	/**
	 * Apply tax exemption fields from XML using mapper.
	 *
	 * @param \Vtiger_Record_Model $record The invoice record
	 *
	 * @return void
	 */
	private function applyTaxExemption(\Vtiger_Record_Model $record): void
	{
		$zwolnienieNode = $this->builder->getXml()->xpath('//Fa/Adnotacje/Zwolnienie');
		if (empty($zwolnienieNode)) {
			return;
		}

		$p19n = $this->builder->getOptionalString('//Fa/Adnotacje/Zwolnienie/P_19N');
		if ('1' === $p19n) {
			return;
		}

		$p19 = $this->builder->getOptionalString('//Fa/Adnotacje/Zwolnienie/P_19');
		if ('1' !== $p19) {
			return;
		}

		$p19a = $this->builder->getOptionalString('//Fa/Adnotacje/Zwolnienie/P_19A');
		if ($p19a) {
			$this->builder->setFieldIfMapped($record, '//Fa/Adnotacje/Zwolnienie/P_19', 'PLL_ACT');
			$this->builder->setFieldIfMapped($record, '//Fa/Adnotacje/Zwolnienie/P_19A', $p19a);
			return;
		}

		$p19b = $this->builder->getOptionalString('//Fa/Adnotacje/Zwolnienie/P_19B');
		if ($p19b) {
			$this->builder->setFieldIfMapped($record, '//Fa/Adnotacje/Zwolnienie/P_19', 'PLL_DIRECTIVE');
			$this->builder->setFieldIfMapped($record, '//Fa/Adnotacje/Zwolnienie/P_19B', $p19b);
			return;
		}

		$p19c = $this->builder->getOptionalString('//Fa/Adnotacje/Zwolnienie/P_19C');
		if ($p19c) {
			$this->builder->setFieldIfMapped($record, '//Fa/Adnotacje/Zwolnienie/P_19', 'PLL_OTHER');
			$this->builder->setFieldIfMapped($record, '//Fa/Adnotacje/Zwolnienie/P_19C', $p19c);
		}
	}
}
