<?php

/**
 * KSeF Correction Invoice Rule class file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Michał Stancelewski <m.stancelewski@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Integrations\KSeF\Service\Reader\RecordBuilder\Rules;

use App\Integrations\KSeF\Service\Reader\RecordBuilder\AbstractRecordBuilder;
use App\QueryGenerator;

/**
 * KSeF Correction Invoice Rule class.
 *
 * Handles correction invoices (KOR, KOR_ZAL, KOR_ROZ):
 * - Sets PrzyczynaKorekty (correction reason) in invoice description
 * - Maps TypKorekty to correction_type picklist
 * - Creates relation to corrected invoice via yf_corrected_invoice field
 * - Filters out lines with StanPrzed=1 (previous state lines)
 */
final class CorrectionRule implements RuleInterface
{
	private AbstractRecordBuilder $builder;
	private ?string $invoiceType = null;

	/** {@inheritDoc} */
	public function __construct(AbstractRecordBuilder $builder)
	{
		$this->builder = $builder;
	}

	/** {@inheritDoc} */
	public function apply(
		\Vtiger_Record_Model $record,
		array $context = [],
	): void {
		$this->invoiceType = $this->builder->getOptionalString('//Fa/RodzajFaktury');

		if (!\in_array($this->invoiceType, ['KOR', 'KOR_ZAL', 'KOR_ROZ'])) {
			return;
		}

		$correctionReason = $this->builder->getOptionalString('//Fa/PrzyczynaKorekty');
		if ($correctionReason) {
			$this->builder->setFieldIfMapped($record, '//Fa/PrzyczynaKorekty', $correctionReason);
		}

		$correctionType = $this->builder->getOptionalString('//Fa/TypKorekty');
		if ($correctionType) {
			$mappedType = match ($correctionType) {
				'1' => 'PLL_ORIGINAL',
				'2' => 'PLL_CORRECTIVE',
				'3' => 'PLL_OTHER',
				default => null,
			};

			if ($mappedType) {
				$fieldName = $this->builder->getMapperService()->getMapper()->get('//Fa/TypKorekty')?->getField();
				if ($fieldName) {
					$record->set($fieldName, $mappedType);
				}
			}
		}

		$this->createCorrectedInvoiceRelation($record);
	}

	/**
	 * Create relation to corrected invoice after invoice is saved.
	 *
	 * This must be called after the invoice has been saved and has an ID.
	 *
	 * @param \Vtiger_Record_Model $recordModel The saved invoice record
	 *
	 * @return void
	 */
	public function createCorrectedInvoiceRelation(\Vtiger_Record_Model $recordModel): void
	{
		if (!$this->builder->getXml()) {
			return;
		}

		$correctedInvoiceId = $this->findCorrectedInvoice();
		if ($correctedInvoiceId) {
			$recordModel->ext['relations'][] = [
				'relatedModule' => $recordModel->getModuleName(),
				'relatedRecords' => $correctedInvoiceId,
			];
		}
	}

	/**
	 * Check if line should be excluded (has StanPrzed=1).
	 *
	 * @param \SimpleXMLElement $lineNode
	 *
	 * @return bool True if line should be excluded
	 */
	public function shouldExcludeLine(\SimpleXMLElement $lineNode): bool
	{
		if (!\in_array($this->invoiceType, ['KOR', 'KOR_ZAL', 'KOR_ROZ'])) {
			return false;
		}

		$stateBefore = $this->builder->getOptionalString('StanPrzed', $lineNode);

		return '1' === $stateBefore;
	}

	/**
	 * Find corrected invoice by KSeF number or invoice number + date + seller VAT.
	 *
	 * @return int|null Corrected invoice ID
	 */
	private function findCorrectedInvoice(): ?int
	{
		$ksefNumber = $this->builder->getOptionalString('//Fa/DaneFaKorygowanej/NrKSeFFaKorygowanej');
		if ($ksefNumber) {
			$invoiceId = $this->findInvoiceByKsefNumber($ksefNumber);
			if ($invoiceId) {
				return $invoiceId;
			}
		}

		$invoiceNumber = $this->builder->getOptionalString('//Fa/DaneFaKorygowanej/NrFaKorygowanej');
		$issueDate = $this->builder->getOptionalString('//Fa/DaneFaKorygowanej/DataWystFaKorygowanej');
		$sellerVatId = $this->builder->getOptionalString('//Podmiot1/DaneIdentyfikacyjne/NIP');

		if ($invoiceNumber && $issueDate && $sellerVatId) {
			return $this->findInvoiceByNumberDateVat($invoiceNumber, $issueDate, $sellerVatId);
		}

		return null;
	}

	/**
	 * Find invoice by KSeF number.
	 *
	 * @param string $ksefNumber
	 *
	 * @return int|null
	 */
	private function findInvoiceByKsefNumber(string $ksefNumber): ?int
	{
		$fieldName = $this->builder->getMapperService()->getMapper()->get('yf_ksef_number')?->getField();
		if (!$fieldName) {
			return null;
		}

		$queryGenerator = new QueryGenerator('FInvoiceCost');
		$queryGenerator->permissions = false;
		$queryGenerator->setFields(['id']);
		$queryGenerator->addCondition($fieldName, $ksefNumber, 'e');

		return $queryGenerator->createQuery()->scalar() ?: null;
	}

	/**
	 * Find invoice by number, date and seller VAT ID.
	 *
	 * @param string $invoiceNumber
	 * @param string $issueDate
	 * @param string $sellerVatId
	 *
	 * @return int|null
	 */
	private function findInvoiceByNumberDateVat(string $invoiceNumber, string $issueDate, string $sellerVatId): ?int
	{
		$sellerVatFieldName = $this->builder->getMapperService()->getMapper()->get('//Podmiot1/DaneIdentyfikacyjne/NIP')?->getField();

		$queryGenerator = new QueryGenerator('FInvoiceCost');
		$queryGenerator->permissions = false;
		$queryGenerator->setFields(['id']);
		$queryGenerator->addCondition('subject', $invoiceNumber, 'e');
		$queryGenerator->addCondition('issue_time', $issueDate, 'e');

		if ($sellerVatFieldName) {
			$queryGenerator->addCondition($sellerVatFieldName, $sellerVatId, 'e');
		}

		return $queryGenerator->createQuery()->scalar() ?: null;
	}
}
